unit MeBmp;

(*------------------------------------------------------------------------------
    Mandelbrot Set Explorer
    Copyright (C) 2003 Chiaki Nakajima

    This file is part of Mandelbrot Set Explorer [MSE].

    MSE is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    MSE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with MSE; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
------------------------------------------------------------------------------*)

interface

uses
{$IFDEF MSWINDOWS}
  Graphics, Dialogs, ExtCtrls,
{$ENDIF}
{$IFDEF LINUX}
  QGraphics, QDialogs, QExtCtrls,
{$ENDIF}
  SysUtils, Types, Classes;

const
  MaxColors = 256;
  Bmp256Width = 384;
  Bmp256Height = 320;

type
  TRgbQuad = packed record
    rgbBlue: Byte;
    rgbGreen: Byte;
    rgbRed: Byte;
    rgbReserved: Byte;
  end;
  TRgbQuadArray = array [0..65535] of TRgbQuad;
  PRgbQuadArray = ^TRgbQuadArray;
  TBmpFileHeader256 = packed record
    bfType: Word;
    bfSize: Longword;
    bfReserved1: Word;
    bfReserved2: Word;
    BfOffBits: Longword;
  end;
  TBmpInfoHeader256 = packed record
    biSize: Longword;
    biWidth: Longint;
    biHeight: Longint;
    biPlanes: Word;
    biBitCount: Word;
    biCompression: Longword;
    biSizeImage: Longword;
    biXPixPerMeter: Longint;
    biYPixPerMeter: Longint;
    biClrUsed: Longword;
    biClrImportant: Longword;
  end;
  PPalette256 = ^TPalette256;
  TPalette256 = array [0..255] of TRgbQuad;
  PPixels256 = ^TPixels256;
  TPixels256 = array [0..Bmp256Height - 1, 0..Bmp256Width - 1] of Byte;
  TBmp256 = class(TObject)
  private
    { Private }
  public
    { Public }
    Palette: PPalette256;
    Pixels: PPixels256;

    constructor Create; virtual;
    destructor Destroy; override;

    procedure Copy(var _Src: TBmp256); virtual;
    procedure Draw(var _Image: TImage);

    function  LoadFromFile(_FileName: String): Boolean; virtual;
    procedure SaveToFile(_FileName: String); virtual;
  end;
  function  RgbToColor(var _Rgb: TRgbQuad): TColor;
  procedure ColorToRgb(var _Col: TColor; var _Rgb: TRgbQuad);

implementation

constructor TBmp256.Create;
begin
  inherited Create;
  New(Palette);
  New(Pixels);
end;

destructor TBmp256.Destroy;
begin
  Dispose(Pixels);
  Dispose(Palette);
  inherited Destroy;
end;

procedure TBmp256.Copy(var _Src: TBmp256);
begin
  Palette^ := _Src.Palette^;
  Pixels^ := _Src.Pixels^;
end;

procedure TBmp256.Draw(var _Image: TImage);
var
  Px, Py: Integer;
  pDstLine: PRgbQuadArray;
begin
  _Image.Picture.Bitmap.Canvas.Lock;
  for Py := 0 to Bmp256Height - 1 do begin
    pDstLine := _Image.Picture.Bitmap.ScanLine[Bmp256Height - 1 - Py];
    for Px := 0 to Bmp256Width - 1 do
      pDstLine^[Px] := Palette^[Pixels^[Py, Px]];
  end;
  _Image.Picture.Bitmap.Canvas.Unlock;
  _Image.Invalidate;
end;

function TBmp256.LoadFromFile(_FileName: String): Boolean;
var
  BmpFileHeader256: TBmpFileHeader256;
  BmpInfoHeader256: TBmpInfoHeader256;
  Stream: TFileStream;
begin
  Result := False;
  Stream := TFileStream(nil);
  try
    try
      Stream := TFileStream.Create(_FileName, fmOpenRead or fmShareDenyWrite);
      Stream.ReadBuffer(BmpFileHeader256, SizeOf(TBmpFileHeader256));
      if (BmpFileHeader256.bfType <> $4D42) then
        raise EInvalidGraphic.CreateFmt('%s is not a bitmap file.', [_FileName]);
      Stream.ReadBuffer(BmpInfoHeader256, SizeOf(TBmpInfoHeader256));
      with BmpInfoHeader256 do
        if ((biSize <> 40) or (biWidth <> Bmp256Width) or (biHeight <> Bmp256Height) or (biPlanes <> 1)
         or (biBitCount <> 8) or (biCompression <> 0) or ((biClrUsed <> 0) and (biClrUsed <> 256))) then
          raise EInvalidGraphic.CreateFmt('%s can not be used in this application.', [_FileName]);
      Stream.ReadBuffer(Palette^, SizeOf(TPalette256));
      Stream.ReadBuffer(Pixels^, SizeOf(TPixels256));
      Result := True;
    except
      on e:EInvalidGraphic do begin
        MessageDlg(e.Message, mtError, [mbOK], 0);
      end;
      on e:EReadError do begin
        MessageDlg(e.Message, mtError, [mbOK], 0);
      end;
    end;
  finally
    Stream.Free;
  end;
end;

procedure TBmp256.SaveToFile(_FileName: String);
var
  BmpFileHeader256: TBmpFileHeader256;
  BmpInfoHeader256: TBmpInfoHeader256;
  Stream: TFileStream;
begin
  Stream := TFileStream(nil);
  try
    try
      Stream := TFileStream.Create(_FileName, fmCreate);
      with BmpFileHeader256 do begin
        bfType := $4d42;
        bfSize := SizeOf(TBmpFileHeader256) + SizeOf(TBmpInfoHeader256)
          + SizeOf(TPalette256) + SizeOf(TPixels256);
        bfReserved1 := 0;
        bfReserved2 := 0;
        bfOffBits := SizeOf(TBmpFileHeader256) + SizeOf(TBmpInfoHeader256) + SizeOf(TPalette256);
      end;
      Stream.WriteBuffer(BmpFileHeader256, SizeOf(TBmpFileHeader256));
      with BmpInfoHeader256 do begin
        biSize := SizeOf(TBmpInfoHeader256);
        biWidth := Bmp256Width;
        biHeight := Bmp256Height;
        biPlanes := 1;
        biBitCount := 8;
        biCompression := 0;
        biSizeImage := SizeOf(TPixels256);
        biXPixPerMeter := 0;
        biYPixPerMeter := 0;
        biClrUsed := 0;
        biClrImportant := 0;
      end;
      Stream.WriteBuffer(BmpInfoHeader256, SizeOf(TBmpInfoHeader256));
      Stream.WriteBuffer(Palette^, SizeOf(TPalette256));
      Stream.WriteBuffer(Pixels^, SizeOf(TPixels256));
    except
      on e:EWriteError do begin
        MessageDlg(e.Message, mtError, [mbOK], 0);
      end;
    end;
  finally
    Stream.Free;
  end;
end;
  // type錾packed recordɂȂƁAO[hEɂĂ܂̂ŁASizeOf֐ԈlƂȂB

function RgbToColor(var _Rgb: TRgbQuad): TColor;
begin
  Result := $02000000 + Longword(_Rgb.rgbBlue) shl 16 +
    Longword(_Rgb.rgbGreen) shl 8 + Longword(_Rgb.rgbRed);
end;

procedure ColorToRgb(var _Col: TColor; var _Rgb: TRgbQuad);
begin
  _Rgb.rgbBlue := Lo(_Col shr 16);
  _Rgb.rgbGreen := Lo(_Col shr 8);
  _Rgb.rgbRed := Lo(_Col);
  _Rgb.rgbReserved := $00;
end;

end.
