unit MeMain;

(*------------------------------------------------------------------------------
    Mandelbrot Set Explorer
    Copyright (C) 2003 Chiaki Nakajima

    This file is part of Mandelbrot Set Explorer [MSE].

    MSE is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    MSE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with MSE; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
------------------------------------------------------------------------------*)

interface

uses
{$IFDEF MSWINDOWS}
  Graphics, Controls, Forms, Dialogs, StdCtrls, ExtCtrls, Menus, ComCtrls, Buttons,
{$ENDIF}
{$IFDEF LINUX}
  QTypes, QGraphics, QControls, QForms, QDialogs, QStdCtrls, QExtCtrls, QMenus, QComCtrls, QButtons,
{$ENDIF}
  SysUtils, Types, Classes, Variants, MeBmp;

const
  VersionString = '1.00';
  SectionMainForm = 'MainForm';
  ImageBmpWidth   = MeBmp.Bmp256Width;
  ImageBmpHeight  = MeBmp.Bmp256Height;
  MapBmpBytes   = ImageBmpWidth * ImageBmpHeight;
  PalMax  = MeBmp.MaxColors;
  MaxOfMaxIter = 5000;
  MaxChildren = 10;
  StringLeft = 'Left';
  StringTop = 'Top';
  StringWidth = 'Width';
  StringHeight = 'Height';
  DefaultRgb0: TRgbQuad = (RgbBlue: $00; RgbGreen: $80; RgbRed: $00; RgbReserved: $00);
  DefaultRgbL: TRgbQuad = (RgbBlue: $80; RgbGreen: $00; RgbRed: $00; RgbReserved: $00);
  DefaultRgbS1L: TRgbQuad = (RgbBlue: $00; RgbGreen: $00; RgbRed: $ff; RgbReserved: $00);
  DefaultRgbS1U: TRgbQuad = (RgbBlue: $00; RgbGreen: $80; RgbRed: $00; RgbReserved: $00);
  DefaultRgbU: TRgbQuad = (RgbBlue: $ff; RgbGreen: $ff; RgbRed: $ff; RgbReserved: $00);
  DefaultRgb255: TRgbQuad = (RgbBlue: $00; RgbGreen: $00; RgbRed: $00; RgbReserved: $00);
  DefaultRgbS1Index: Integer = 40;

type
  TFloat = Extended;
  TMapList = (NoMap, EntireMap, WideMap, CurrentMap);
  TMainModeList = (mmNotInitialized, mmInitialize, mmNotCommanded, mmAccessMapFile,
    mmSetupPrinter, mmPrint, mmEditTitle, mmEditMaxIter, mmDeleteZoomBox,
    mmShowResizeBox, mmDragResizeBox, mmShowZoomBox, mmSelectedZoomBox,
    mmZoomIn, mmZoomOut, mmJump, mmEditPalette, mmChangeFont, mmHelpIntroduction, mmHelpMode);
  TScanModeList = (smNotScanning, smScanningEntire, smScanningWide, smScanningCurrent);
  TColorTable = array [0..PalMax-1] of Longword;
  TFormMain = class(TForm)
    PanelClient: TPanel;
    PanelSpeedBar: TPanel;
    PanelStatusBar: TPanel;
    PanelEntire: TPanel;
    PanelWide: TPanel;
    PanelCurrent: TPanel;
    ImageEntire: TImage;
    ImageWide: TImage;
    ImageCurrent: TImage;
    MainMenu: TMainMenu;
    miFile: TMenuItem;
    miFileUpdateMapFile: TMenuItem;
    miFileSeparator1: TMenuItem;
    miFileExit: TMenuItem;
    miEdit: TMenuItem;
    miEditTitle: TMenuItem;
    miEditChangeMaxIteration: TMenuItem;
    miEditDeleteZoomBox: TMenuItem;
    miEditPalette: TMenuItem;
    miEditSeparator1: TMenuItem;
    miEditCopyToClipboard: TMenuItem;
    miZoom: TMenuItem;
    miZoomNewZoomBox: TMenuItem;
    miZoomDisplayZoomBox: TMenuItem;
    miZoomSeparator1: TMenuItem;
    miZoomIn: TMenuItem;
    miZoomSeparator2: TMenuItem;
    miZoomOut: TMenuItem;
    miZoomJumpToEntire: TMenuItem;
    miZoomSeparator3: TMenuItem;
    miZoomAbort: TMenuItem;
    miOption: TMenuItem;
    miOptionDisplayMapWindow: TMenuItem;
    miOptionDisplay3DWindow: TMenuItem;
    miOptionSeparator1: TMenuItem;
    miOptionUseImageFiles: TMenuItem;
    miOptionSeparator2: TMenuItem;
    miOptionSelectFont: TMenuItem;
    miHelp: TMenuItem;
    miHelpIntroduction: TMenuItem;
    miHelpMode: TMenuItem;
    miHelpAbout: TMenuItem;
    SpeedButtonUpdateMapFile: TSpeedButton;
    SpeedButtonEditTitle: TSpeedButton;
    SpeedButtonChangeMaxIteration: TSpeedButton;
    SpeedButtonDeleteZoomBox: TSpeedButton;
    SpeedButtonEditPalette: TSpeedButton;
    SpeedButtonCopyToClipboard: TSpeedButton;
    SpeedButtonNewZoomBox: TSpeedButton;
    SpeedButtonDisplayZoomBox: TSpeedButton;
    SpeedButtonZoomIn: TSpeedButton;
    SpeedButtonZoomOut: TSpeedButton;
    SpeedButtonJumpToEntire: TSpeedButton;
    SpeedButtonAbort: TSpeedButton;
    SpeedButtonDisplayMapWindow: TSpeedButton;
    SpeedButtonDisplay3DWindow: TSpeedButton;
    LabelEntire: TLabel;
    LabelWide: TLabel;
    LabelCurrent: TLabel;
    LabelTitle: TLabel;
    LabelR2: TLabel;
    LabelI2: TLabel;
    LabelR1: TLabel;
    LabelI1: TLabel;
    LabelSpan: TLabel;
    LabelMag: TLabel;
    LabelNestLevel: TLabel;
    LabelMaxIter: TLabel;
    LabelStatus: TLabel;
    PopupMenu: TPopupMenu;
    piNewZoomBox: TMenuItem;
    piDisplayZoomBox: TMenuItem;
    piSeparator1: TMenuItem;
    piZoomIn: TMenuItem;
    piZoomOut: TMenuItem;
    piSeparator2: TMenuItem;
    piJumptoEntire: TMenuItem;
    piSeparator3: TMenuItem;
    piAbort: TMenuItem;
    EntireGuideLineV: TShape;
    EntireGuideLineH: TShape;
    WideGuideBox: TShape;
    ZoomBox1: TShape;
    ZoomBox2: TShape;
    ZoomBox3: TShape;
    ZoomBox4: TShape;
    ZoomBox5: TShape;
    ZoomBox6: TShape;
    ZoomBox7: TShape;
    ZoomBox8: TShape;
    ZoomBox9: TShape;
    ZoomBox10: TShape;
    ResizeBox: TShape;
    ResizeBoxLT: TShape;
    ResizeBoxRT: TShape;
    ResizeBoxLB: TShape;
    ResizeBoxRB: TShape;
    ResizeBoxT: TShape;
    ResizeBoxB: TShape;
    ResizeBoxL: TShape;
    ResizeBoxR: TShape;
    TimerIdle: TTimer;
    EditTitle: TEdit;
    ListBoxMaxIteration: TListBox;
    PanelOkCancel: TPanel;
    ButtonOk: TButton;
    ButtonCancel: TButton;
    FontDialog: TFontDialog;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormActivate(Sender: TObject);
    procedure TimerIdleTimer(Sender: TObject);
    procedure LoadIniFile;
    procedure SaveIniFile;
    procedure Idle;
    procedure ChangeFont(_Font: TFont);
    procedure BeginScan(_ScanMap: TMapList);
    procedure Scan;
    function  ConfirmAbortScan: Boolean;
    procedure ChangeCurrentMapNode(_TreeNode: TTreeNode);
    procedure MapImageClear(_ScanMap: TMapList);
    procedure SetPalette(_Map: TMapList);
    procedure ChangeCurrentMapPaletteTemporary;
    procedure CurrentMapPaletteChanged;
    function  CalcPixel(var _Cx, _Cy: TFloat; _MaxIter: Word): Word;
    function  GetImageFileName(_TreeNode: TTreeNode): String;
    function  MapNode(_Map: TMapList): TTreeNode;
    function  NotCommanded: Boolean;
    function  Scanning: Boolean;
    procedure UpdateCurrentLabels;
    procedure InitializeEntire;
    procedure ZoomIn(_TreeNode: TTreeNode);
    procedure ZoomOut;
    procedure JumpTree(_TreeNode: TTreeNode);
    procedure ChangeMainMode(_Mode: TMainModeList);
    procedure ChangeScanMode(_Mode: TScanModeList);
    procedure CancelCommand;
    procedure UpdateMainMenu;
    procedure UpdatePopupMenu;
    procedure UpdateSpeedButton;
    procedure AdjustAspect(var W, H: Integer);
    function  GetAspectH(W: Integer): Integer;
    function  GetAspectW(H: Integer): Integer;
    procedure ResizeResizeBox;
    procedure ShowResizeBox;
    procedure HideResizeBox;
    procedure ResizeBoxClipT(var TmpW, TmpH: Integer);
    procedure ResizeBoxClipL(var TmpW, TmpH: Integer);
    procedure ResizeBoxClipR(var TmpW, TmpH: Integer);
    procedure ResizeBoxClipB(var TmpW, TmpH: Integer);
    procedure ResizeBoxMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure ResizeBoxMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure ResizeBoxMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure ResizeBoxLTMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure ResizeBoxLBMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure ResizeBoxRTMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure ResizeBoxRBMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure ResizeBoxTMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure ResizeBoxLMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure ResizeBoxRMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure ResizeBoxBMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure GetZoomBoxRect(_Client, _Child: TTreeNode; _Image: TImage; var _ResultRect: TRect);
    procedure ShowZoomBox;
    procedure HideZoomBox;
    procedure ZoomBoxMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure ImageCurrentClick(Sender: TObject);
    procedure ShowGuideEntire;
    procedure HideGuideEntire;
    procedure ShowGuideWide;
    procedure HideGuideWide;
    procedure miMenuClick(Sender: TObject);
    procedure miFileUpdateMapFileClick(Sender: TObject);
    procedure miFileExitClick(Sender: TObject);
    procedure miZoomNewZoomBoxClick(Sender: TObject);
    procedure miZoomDisplayZoomBoxClick(Sender: TObject);
    procedure miZoomInClick(Sender: TObject);
    procedure miZoomOutClick(Sender: TObject);
    procedure miZoomJumpToEntireClick(Sender: TObject);
    procedure miZoomAbortClick(Sender: TObject);
    procedure miEditTitleClick(Sender: TObject);
    procedure miEditChangeMaxIterationClick(Sender: TObject);
    procedure miEditDeleteZoomBoxClick(Sender: TObject);
    procedure miEditPaletteClick(Sender: TObject);
    procedure miEditCopyToClipboardClick(Sender: TObject);
    procedure miOptionDisplayMapWindowClick(Sender: TObject);
    procedure miOptionDisplay3DWindowClick(Sender: TObject);
    procedure miOptionUseImageFilesClick(Sender: TObject);
    procedure miOptionSelectFontClick(Sender: TObject);
    procedure miHelpIntroductionClick(Sender: TObject);
    procedure miHelpModeClick(Sender: TObject);
    procedure miHelpAboutClick(Sender: TObject);
    procedure PopupMenuPopup(Sender: TObject);
    procedure ButtonOkClick(Sender: TObject);
    procedure ButtonCancelClick(Sender: TObject);
    procedure EditTitleKeyPress(Sender: TObject; var Key: Char);
    procedure ListBoxMaxIterationKeyPress(Sender: TObject; var Key: Char);
    procedure ClickToHelp(Sender: TObject);
  private
    { Private }
    ScanPy, ScanMaxIter: Integer;
    ScanR1, ScanCy, ScanStep: TFloat;
    PaletteIndex: array [0..MaxOfMaxIter] of Byte;
    ProcessingIdle: Boolean;
    SaveCursor: TCursor;
  public
    { Public }
    MainMode: TMainModeList;
    ScanMode: TScanModeList;
    CurrentMapNode: TTreeNode;
    ImageMap: array [EntireMap..CurrentMap] of TImage;
    ScanCalculation: Boolean;
    RequestToAbortScan: Boolean;
    DraggingOrgPoint: TPoint;
    SelectedZoomBoxIndex: Integer;
    MapFileName: String;
    ScanEndSoundFileName: String;
    PrintScale: Single;
    ZoomBox: array [1..MaxChildren] of TShape;
    ZoomBoxNode: array [1..MaxChildren] of TTreeNode;
    ImageBmp256: array [EntireMap..CurrentMap] of TBmp256;
  end;

var
  FormMain: TFormMain;

implementation

uses
{$IFDEF MSWINDOWS}
  Clipbrd,
{$ENDIF}
{$IFDEF LINUX}
  QClipbrd,
{$ENDIF}
  MePalEdt, MeView3D, MeZoomIn, MeTree, MeAbout, MeClose, MeIni, MeHelp, MeIntro;

const
  DefaultMapFileName = 'Mse.map';

  MapTitlePrefix = 'Map';
  ImageFilePrefix = 'Map';
  ImageFileExtention = '.bmp';
  StringZoomPrmDlg = 'ZoomPrmDlg';
  StringView3DDlg = 'View3DDlg';
  StringImageFile = 'ImageFile';
  StringFontName = 'FontName';
  StringFontSize = 'FontSize';
  StringFixedFontName = 'FixedFontName';
  StringFixedFontSize = 'FixedFontSize';
  SqrBdr  = 100.0;
  MaxIterItems = 8;
  MaxIterList : array [0..MaxIterItems - 1] of Integer
    = (100, 200, 300, 500, 1000, 2000, 3000, MaxOfMaxIter);

{$IFDEF MSWINDOWS}
{$R *.dfm}
{$ENDIF}
{$IFDEF LINUX}
{$R *.xfm}
{$ENDIF}

procedure TFormMain.FormCreate(Sender: TObject);

var
  TmpMap: TMapList;
  Bmp: TBitmap;
  TmpI: Integer;
  TmpList: TStringList;
begin
  MainMode := mmNotInitialized;
  ScanMode := smNotScanning;
  ImageMap[EntireMap] := ImageEntire;
  ImageMap[WideMap] := ImageWide;
  ImageMap[CurrentMap] := ImageCurrent;
  CurrentMapNode := nil;
  for TmpMap := EntireMap to CurrentMap do begin
    Bmp := nil;
    try
      Bmp := TBitmap.Create;
      Bmp.PixelFormat := pf32bit;
      Bmp.Width := ImageBmpWidth;
      Bmp.Height := ImageBmpHeight;
      ImageMap[TmpMap].Picture.Graphic := Bmp;
    finally
      Bmp.Free;
    end;
    ImageBmp256[TmpMap] := TBmp256.Create;
    MapImageClear(TmpMap);
  end;
  TmpList := TStringList.Create;
  try
    for TmpI := 0 to MaxIterItems - 1 do
      TmpList.Add(Format('%5d', [MaxIterList[TmpI]]));
    ListBoxMaxIteration.Items := TmpList;
  finally
    TmpList.Free;
  end;
  ScanCalculation := False;
  RequestToAbortScan := False;
  SelectedZoomBoxIndex := 0;
  for TmpI := 1 to MaxChildren do
    ZoomBoxNode[TmpI] := nil;
  MapFileName := ExtractFilePath(Application.ExeName) + DefaultMapFileName;
  PrintScale := 1.0;
  ZoomBox[ 1] := ZoomBox1;
  ZoomBox[ 2] := ZoomBox2;
  ZoomBox[ 3] := ZoomBox3;
  ZoomBox[ 4] := ZoomBox4;
  ZoomBox[ 5] := ZoomBox5;
  ZoomBox[ 6] := ZoomBox6;
  ZoomBox[ 7] := ZoomBox7;
  ZoomBox[ 8] := ZoomBox8;
  ZoomBox[ 9] := ZoomBox9;
  ZoomBox[10] := ZoomBox10;
  ProcessingIdle := False;
  TimerIdle.Enabled := True;
{$IFDEF LINUX}
  FormMain.WindowState := wsMinimized;
{$ENDIF}
    // This comment part is written in Japanese. Sorry...
    // Nov. 11, 203
    // čŏŏĂAShow\bhōĕ\悤ɂȂ,
    // Left, TopvpeBݒ肳ȂB
    // Init@C̓ǂݍ݂ɂvpeBݒŁAƂΐ0,0ɕ\ɂ
    // 炸AvpeBǂݏoLeft=4, Top=23ԂĂB
    // xEChEhbOƐlǂݏo悤ɂȂ̂ŁA
    // EBhE}l[WƂƊ֌ŴȂB
    // ̌ۂ́AEBhËʒuwSetBoundsgĂPȂB
end;

procedure TFormMain.FormClose(Sender: TObject; var Action: TCloseAction);
var
  TmpMap: TMapList;
begin
  for TmpMap := EntireMap to CurrentMap do begin
    ImageBmp256[TmpMap].Destroy;
    ImageMap[TmpMap].Free;
  end;
end;

procedure TFormMain.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
var
  FileNameEntire: String;
begin
  TimerIdle.Enabled := False;
  CanClose := (CloseQueryDlg.ShowModal = mrOk);
  if (CanClose) then begin
    SaveIniFile;
    FileNameEntire := GetImageFileName(MapNode(EntireMap));
    if (not FileExists(FileNameEntire)) then
      ImageBmp256[EntireMap].SaveToFile(FileNameEntire);
    ZoomPrmDlg.SaveMapFile(MapFileName);
    UpdateMainMenu;
    UpdatePopupMenu;
    UpdateSpeedButton;
    ZoomPrmDlg.FreeZoomPrm(MapNode(EntireMap));
    MapNode(EntireMap).Delete;
  end
  else
    TimerIdle.Enabled := True;
end;

procedure TFormMain.FormActivate(Sender: TObject);

begin
  if (MainMode = mmNotInitialized) then Exit;
{$IFDEF LINUX}
  if (miOptionDisplayMapWindow.Checked) then
    ZoomPrmDlg.Show;
  if (miOptionDisplay3DWindow.Checked) then
    View3DDlg.Show;
{$ENDIF}
end;

procedure TFormMain.TimerIdleTimer(Sender: TObject);

begin
  Idle;
end;

procedure TFormMain.LoadIniFile;

var
  TmpFont: TFont;
begin
  ZoomPrmDlg.LoadIniFile;
  View3DDlg.LoadIniFile;
  miOptionDisplayMapWindow.Checked := Ini.ReadBool(SectionMainForm, StringZoomPrmDlg, True);
  ZoomPrmDlg.Visible := miOptionDisplayMapWindow.Checked;
  miOptionDisplay3DWindow.Checked := Ini.ReadBool(SectionMainForm, StringView3DDlg, True);
  View3DDlg.Visible := miOptionDisplay3DWindow.Checked;
  miOptionUseImageFiles.Checked := Ini.ReadBool(SectionMainForm, StringImageFile, True);
  Left := Ini.ReadInteger(SectionMainForm, StringLeft, 0);
  Top := Ini.ReadInteger(SectionMainForm, StringTop, 0);
  TmpFont := FormMain.Font;
{$IFDEF MSWINDOWS}
  PanelClient.Font.Name := Ini.ReadString(SectionMainForm, StringFixedFontName, 'lr SVbN');
  PanelClient.Font.Size := Ini.ReadInteger(SectionMainForm, StringFixedFontSize, 10);
  TmpFont.Name := Ini.ReadString(SectionMainForm, StringFontName, 'lr oSVbN');
  TmpFont.Size := Ini.ReadInteger(SectionMainForm, StringFontSize, 10);
{$ENDIF}
{$IFDEF LINUX}
  PanelClient.Font.Name := Ini.ReadString(SectionMainForm, StringFixedFontName, 'helvetica');
  PanelClient.Font.Size := Ini.ReadInteger(SectionMainForm, StringFixedFontSize, 10);
  TmpFont.Name := Ini.ReadString(SectionMainForm, StringFontName, 'helvetica');
  TmpFont.Size := Ini.ReadInteger(SectionMainForm, StringFontSize, 10);
{$ENDIF}
  ChangeFont(TmpFont);
end;

procedure TFormMain.SaveIniFile;
begin
  Ini.WriteInteger(SectionMainForm, StringFontSize, FormMain.Font.Size);
  Ini.WriteString(SectionMainForm, StringFontName, FormMain.Font.Name);
  Ini.WriteInteger(SectionMainForm, StringFixedFontSize, PanelClient.Font.Size);
  Ini.WriteString(SectionMainForm, StringFixedFontName, PanelClient.Font.Name);
  Ini.WriteInteger(SectionMainForm, StringLeft, Left);
  Ini.WriteInteger(SectionMainForm, StringTop, Top);
  Ini.WriteBool(SectionMainForm, StringZoomPrmDlg, miOptionDisplayMapWindow.Checked);
  Ini.WriteBool(SectionMainForm, StringView3DDlg, miOptionDisplay3DWindow.Checked);
  Ini.WriteBool(SectionMainForm, StringImageFile, miOptionUseImageFiles.Checked);
  View3DDlg.SaveIniFile;
  ZoomPrmDlg.SaveIniFile;
end;

procedure TFormMain.Idle;

begin
  if (ProcessingIdle) then Exit;
  ProcessingIdle := True;
  if (MainMode = mmNotInitialized) then
    InitializeEntire;
  if (Scanning) then
    Scan;
  ProcessingIdle := False;
end;

procedure TFormMain.ChangeFont(_Font: TFont);
begin
  FormMain.Font.Name := _Font.Name;
  FormMain.Font.Size := _Font.Size;
  EditTitle.Font.Name := _Font.Name;
  ZoomPrmDlg.Font.Name := _Font.Name;
  ZoomPrmDlg.Font.Size := _Font.Size;
  HelpIntroDlg.Font.Name := _Font.Name;
  HelpIntroDlg.Font.Size := _Font.Size;
  HelpModeDlg.Font.Name := _Font.Name;
  HelpModeDlg.Font.Size := _Font.Size;
  FormMain.Invalidate;
  if (ZoomPrmDlg.Visible) then
    ZoomPrmDlg.Invalidate;
end;

function TFormMain.GetImageFileName(_TreeNode: TTreeNode): String;
begin
  Result := ExtractFilePath(Application.ExeName) + ImageFilePrefix
    + IntToStr(TZoomPrm(_TreeNode.Data).SerialNo) + ImageFileExtention;
end;

procedure TFormMain.SetPalette(_Map: TMapList);
var
  TmpI, xOver, DifR, DifG, DifB, BaseR, BaseG, BaseB: Integer;

  function MulDiv(_A, _B, _C: Integer): Integer;
  begin
    Result := (_A * _B) div _C;
  end;

begin
  with ImageBmp256[_Map] do begin
    Palette^[0] := DefaultRgb0;
    with TZoomPrm(MapNode(_Map).Data) do begin
      xOver := RgbS1index;
      BaseR := Integer(RgbL.rgbRed);
      BaseG := Integer(RgbL.rgbGreen);
      BaseB := Integer(RgbL.rgbBlue);
      DifR := Integer(RgbS1L.rgbRed) - BaseR;
      DifG := Integer(RgbS1L.rgbGreen) - BaseG;
      DifB := Integer(RgbS1L.rgbBlue) - BaseB;
    end;
    for TmpI := 1 to xOver - 1 do begin
      Palette^[TmpI].rgbReserved := $00;
      Palette^[TmpI].rgbRed := Lo(MulDiv(DifR, TmpI, xOver - 1) + BaseR);
      Palette^[TmpI].rgbGreen := Lo(MulDiv(DifG, TmpI, xOver - 1) + BaseG);
      Palette^[TmpI].rgbBlue := Lo(MulDiv(DifB, TmpI, xOver - 1) + BaseB);
    end;
    with TZoomPrm(MapNode(_Map).Data) do begin
      BaseR := Integer(RgbS1U.rgbRed);
      BaseG := Integer(RgbS1U.rgbGreen);
      BaseB := Integer(RgbS1U.rgbBlue);
      DifR := Integer(RgbU.rgbRed) - BaseR;
      DifG := Integer(RgbU.rgbGreen) - BaseG;
      DifB := Integer(RgbU.rgbBlue) - BaseB;
    end;
    for TmpI := 0 to PalMax - 2 - xOver do begin
      Palette^[TmpI + xOver].rgbReserved := $00;
      Palette^[TmpI + xOver].rgbRed := Lo(MulDiv(DifR, TmpI, PalMax - 2 - xOver) + BaseR);
      Palette^[TmpI + xOver].rgbGreen := Lo(MulDiv(DifG, TmpI, PalMax - 2 - xOver) + BaseG);
      Palette^[TmpI + xOver].rgbBlue := Lo(MulDiv(DifB, TmpI, PalMax - 2 - xOver) + BaseB);
    end;
    Palette^[PalMax - 1] := DefaultRgb255;
  end;
end;

procedure TFormMain.ChangeCurrentMapPaletteTemporary;
begin
  SetPalette(CurrentMap);
  ImageBmp256[CurrentMap].Draw(ImageMap[CurrentMap]);
  View3DDlg.SrcBmpChanged;
end;

procedure TFormMain.CurrentMapPaletteChanged;
begin
  SetPalette(CurrentMap);
  ImageBmp256[CurrentMap].Draw(ImageMap[CurrentMap]);
  if (MapNode(CurrentMap) = MapNode(EntireMap)) then begin
    ImageBmp256[EntireMap].Copy(ImageBmp256[CurrentMap]);
    ImageBmp256[EntireMap].Draw(ImageMap[EntireMap]);
  end;
  if (MapNode(CurrentMap) = MapNode(WideMap)) then begin
    ImageBmp256[WideMap].Copy(ImageBmp256[CurrentMap]);
    ImageBmp256[WideMap].Draw(ImageMap[WideMap]);
  end;
  View3DDlg.SrcBmpChanged;
  if (miOptionUseImageFiles.Checked) then
    ImageBmp256[CurrentMap].SaveToFile(GetImageFileName(MapNode(CurrentMap)));
end;

function TFormMain.CalcPixel(var _Cx, _Cy: TFloat; _MaxIter: Word): Word;
var
  R, I, SqrR, SqrI: TFloat;
  N: Word;
begin
  R := 0.0;
  I := 0.0;
  SqrR := 0.0;
  SqrI := 0.0;
  N := 0;
  while (((SqrR + SqrI) < SqrBdr) and (N < _MaxIter)) do begin
    I := 2.0 * R * I + _Cy;
    R := SqrR - SqrI + _Cx;
    SqrR := Sqr(R);
    SqrI := Sqr(I);
    Inc(N);
  end;
  Result := N;
end;

procedure TFormMain.MapImageClear(_ScanMap: TMapList);
var
  Px, Py: Integer;
  pDstLine: PRgbQuadArray;
  BackColor: TRgbQuad;
begin
  ImageMap[_ScanMap].Picture.Bitmap.Canvas.Lock;
  BackColor := DefaultRgb0;
  for Py := 0 to ImageBmpHeight - 1 do begin
    pDstLine := ImageMap[_ScanMap].Picture.Bitmap.ScanLine[Py];
    for Px := 0 to ImageBmpWidth - 1 do
      pDstLine^[Px] := BackColor;
  end;
  ImageMap[_ScanMap].Picture.Bitmap.Canvas.UnLock;
  ImageMap[_ScanMap].Invalidate;
end;

procedure TFormMain.BeginScan(_ScanMap: TMapList);
var
  ImageFileName: String;
  TmpI: Integer;
begin
  ScanCalculation := False;
  case _ScanMap of
    EntireMap: begin
      ChangeScanMode(smScanningEntire);
      HideGuideEntire;
      if (MainMode = mmInitialize) then begin
        MapImageClear(_ScanMap);
        ScanCalculation := True;
      end;
    end;
    WideMap: begin
      ChangeScanMode(smScanningWide);
      ShowGuideEntire;
      HideGuideWide;
      if (MapNode(WideMap) = MapNode(EntireMap)) then begin
        ImageBmp256[WideMap].Copy(ImageBmp256[EntireMap]);
        ImageBmp256[WideMap].Draw(ImageMap[WideMap]);
      end
      else begin
        MapImageClear(_ScanMap);
        ScanCalculation := (MapNode(WideMap) <> nil);
      end;
    end;
    CurrentMap: begin
      ChangeScanMode(smScanningCurrent);
      ShowGuideEntire;
      ShowGuideWide;
      UpdateCurrentLabels;
      if ((MainMode <> mmEditMaxIter)
        and (MapNode(CurrentMap) = MapNode(EntireMap))) then begin
        ImageBmp256[CurrentMap].Copy(ImageBmp256[EntireMap]);
        ImageBmp256[CurrentMap].Draw(ImageMap[CurrentMap]);
      end
      else begin
        MapImageClear(_ScanMap);
        ScanCalculation := True;
      end;
    end;
  end;
  if (ScanCalculation) then begin
    ImageFileName := GetImageFileName(MapNode(_ScanMap));
    if (FileExists(ImageFileName)) then begin
      ImageBmp256[_ScanMap].LoadFromFile(ImageFileName);
      ImageBmp256[_ScanMap].Draw(ImageMap[_ScanMap]);
      ScanCalculation := False;
    end;
  end;
  if (ScanCalculation) then begin
    with TZoomPrm(MapNode(_ScanMap).Data) do begin
      ScanMaxIter := MaxIter;
      ScanR1 := R1;
      ScanCy := I1;
      ScanStep := Span / ImageBmpWidth;
      for TmpI := 0 to MaxIter do
        PaletteIndex[TmpI] := Lo((Longint(TmpI) * Longint(PalMax - 1)) div MaxIter);
    end;
    SetPalette(_ScanMap);
    SaveCursor := Screen.Cursor;
    Screen.Cursor := crHourGlass;
    ZoomPrmDlg.TreeView.Enabled := False;
    View3DDlg.PanelSwitches.Enabled := False;
  end;
  ScanPy := 0;
end;

procedure TFormMain.Scan;
var
  ScanningMap: TMapList;
  Px: Integer;
  Cx: TFloat;
  PalIndex: Byte;
  pDstLine: PRgbQuadArray;
begin
  ScanningMap := NoMap;
  case ScanMode of
    smNotScanning: ScanningMap := NoMap;
    smScanningEntire: ScanningMap := EntireMap;
    smScanningWide: ScanningMap := WideMap;
    smScanningCurrent: ScanningMap := CurrentMap;
  end;
  if (ScanCalculation) then begin
    ImageMap[ScanningMap].Picture.Bitmap.Canvas.Lock;
    Cx := ScanR1;
    pDstLine := ImageMap[ScanningMap].Picture.Bitmap.ScanLine[ImageBmpHeight - 1 - ScanPy];
    for Px := 0 to ImageBmpWidth - 1 do begin
      PalIndex := PaletteIndex[CalcPixel(Cx, ScanCy, ScanMaxIter)];
      ImageBmp256[ScanningMap].Pixels^[ScanPy, Px] := PalIndex;
      pDstLine^[Px] := ImageBmp256[ScanningMap].Palette^[PalIndex];
      Cx := Cx + ScanStep;
      if (RequestToAbortScan) then begin
        ScanCalculation := False;
        RequestToAbortScan := False;
        ChangeScanMode(smNotScanning);
        Break;
      end;
    end;
    ImageMap[ScanningMap].Picture.Bitmap.Canvas.Unlock;
    ScanCy := ScanCy + ScanStep;
  end
  else
    ScanPy := ImageBmpHeight - 1;
  ImageMap[ScanningMap].Invalidate;
  Inc(ScanPy);
  if (ScanPy > ImageBmpHeight - 1) then begin
    if (ScanCalculation) then begin
      if (miOptionUseImageFiles.Checked) then
        ImageBmp256[ScanningMap].SaveToFile(GetImageFileName(MapNode(ScanningMap)));
    end;
    Screen.Cursor := crDefault;
    ZoomPrmDlg.TreeView.Enabled := True;
    View3DDlg.PanelSwitches.Enabled := True;
    ScanCalculation := False;
    RequestToAbortScan := False;
    case ScanMode of
      smScanningEntire: begin
        if (MainMode = mmInitialize) then
          BeginScan(CurrentMap)
        else begin
          ChangeMainMode(mmNotCommanded);
          ChangeScanMode(smNotScanning);
        end;
      end;
      smScanningWide: begin
        ShowGuideWide;
        if (MainMode = mmJump) then
          BeginScan(CurrentMap)
        else begin
          ChangeMainMode(mmNotCommanded);
          ChangeScanMode(smNotScanning);
        end;
      end;
      smScanningCurrent: begin
        if ((MainMode = mmEditMaxIter)
          and (MapNode(CurrentMap) = MapNode(EntireMap))) then begin
          ImageBmp256[EntireMap].Copy(ImageBmp256[CurrentMap]);
          ImageBmp256[EntireMap].Draw(ImageMap[EntireMap]);
        end;
        View3DDlg.SrcBmpChanged;
        ChangeMainMode(mmNotCommanded);
        ChangeScanMode(smNotScanning);
      end;
    end;
  end;
end;

function TFormMain.ConfirmAbortScan: Boolean;
begin
  Result := (MessageDlg('Now drawing. Abort?', mtConfirmation, mbOkCancel, 0) = mrOk);
end;

procedure TFormMain.ChangeCurrentMapNode(_TreeNode: TTreeNode);
begin
  if (_TreeNode = nil) then Exit;
 {ZoomPrmDlg.Outline.Items[_TreeIndex].FullExpand;}
  CurrentMapNode := _TreeNode;
  ZoomPrmDlg.TreeView.Selected := _TreeNode;
end;

function TFormMain.MapNode(_Map: TMapList): TTreeNode;
begin
  Result := nil;
  case _Map of
    EntireMap: Result := ZoomPrmDlg.TreeView.Items[0];
    WideMap: Result := CurrentMapNode.Parent;
    CurrentMap: Result := CurrentMapNode;
  end;
end;

function TFormMain.Scanning: Boolean;
begin
  Result := (ScanMode <> smNotScanning);
end;

function TFormMain.NotCommanded: Boolean;
begin
  Result := (MainMode = mmNotCommanded);
end;

procedure TFormMain.UpdateCurrentLabels;
var
  TmpS: String;
begin
  with TZoomPrm(MapNode(CurrentMap).Data) do begin
    TmpS := MapNode(CurrentMap).Text;
    if (Length(TmpS) > 12) then
      TmpS := Format('%.14s', [TmpS]) + '...';
    LabelTitle.Caption := 'Title' + '[' + TmpS + ']';
    LabelR1.Caption := Format('r1[%19.16f]', [R1]);
    LabelI1.Caption := Format('i1[%19.16f]', [I1]);
    LabelR2.Caption := Format('r2[%19.16f]', [R1 + Span]);
    LabelI2.Caption := Format('i2[%19.16f]', [I1 + (Span / ImageMap[CurrentMap].Width * ImageMap[CurrentMap].Height)]);
    LabelSpan.Caption := Format('Span[%19.17f]', [Span]);
    LabelMag.Caption := Format('Mag.[%19.1n]', [TZoomPrm(MapNode(EntireMap).Data).Span / Span]);
    LabelNestLevel.Caption := Format('Nest Level[%2d]', [MapNode(CurrentMap).Level]);
    LabelMaxIter.Caption := Format('MaxIter.[%4d]', [MaxIter]);
  end;
end;

procedure TFormMain.InitializeEntire;
var
  ZoomPrm: TZoomPrm;
begin
  ZoomPrm := TZoomPrm.Create;
  try
    with ZoomPrm do begin
      R1 := -2.2;
      I1 := -1.2;
      Span := 3.0;
      MaxIter := 100;
      SerialNo := -1;
      RgbL := DefaultRgbL;
      RgbS1L := DefaultRgbS1L;
      RgbS1U := DefaultRgbS1U;
      RgbU := DefaultRgbU;
      RgbS1Index := DefaultRgbS1Index;
    end;
    if (FileExists(MapFileName)) then
      ZoomPrmDlg.LoadMapFile(MapFileName)
    else
      ZoomPrmDlg.AddZoomPrm(nil, 'Entire Map', ZoomPrm);
  finally
    ZoomPrm.Free;
  end;
  ChangeCurrentMapNode(MapNode(EntireMap));
  FormMain.WindowState := wsNormal;
  Visible := True;
  ChangeMainMode(mmInitialize);
  BeginScan(EntireMap);
end;

procedure TFormMain.ZoomIn(_TreeNode: TTreeNode);
begin
  ChangeMainMode(mmZoomIn);
  ChangeCurrentMapNode(_TreeNode);
  ImageBmp256[WideMap].Copy(ImageBmp256[CurrentMap]);
  ImageBmp256[WideMap].Draw(ImageMap[WideMap]);
  BeginScan(CurrentMap);
end;

procedure TFormMain.ZoomOut;
begin
  ChangeMainMode(mmZoomOut);
  ChangeCurrentMapNode(MapNode(WideMap));
  ImageBmp256[CurrentMap].Copy(ImageBmp256[WideMap]);
  ImageBmp256[CurrentMap].Draw(ImageMap[CurrentMap]);
  UpdateCurrentLabels;
  View3DDlg.SrcBmpChanged;
  BeginScan(WideMap);
end;

procedure TFormMain.JumpTree(_TreeNode: TTreeNode);
begin
  if (_TreeNode = MapNode(CurrentMap)) then begin
    MessageDlg('The distination is the current map.', mtError, [mbOk], 0);
    exit;
  end;
  ChangeMainMode(mmJump);
  ChangeCurrentMapNode(_TreeNode);
  BeginScan(WideMap);
end;

procedure TFormMain.ChangeMainMode(_Mode: TMainModeList);
begin
  MainMode := _Mode;
  UpdateMainMenu;
  UpdatePopupMenu;
  UpdateSpeedButton;
  case _Mode of
{$IFDEF MSWINDOWS}
    mmNotInitialized  : {do nothing};
    mmInitialize      : LabelStatus.Caption := '.  ҂c';
    mmNotCommanded    : LabelStatus.Caption := '';
    mmAccessMapFile   : LabelStatus.Caption := '}bvt@CɃANZXc';
    mmSetupPrinter    : LabelStatus.Caption := 'v^ݒc';
    mmPrint           : LabelStatus.Caption := 'c';
    mmEditTitle       : LabelStatus.Caption := '^Cg͂AOKĂB';
    mmEditMaxIter     : LabelStatus.Caption := 'ő唽񐔂IOKĂB';
    mmDeleteZoomBox   :{LabelStatus.Caption := ''};
    mmShowResizeBox   : LabelStatus.Caption := 'Y[gݒ肵āAY[ECR}hsĂB';
    mmDragResizeBox   :{LabelStatus.Caption := 'Y[gݒ蒆c'};
    mmShowZoomBox     : LabelStatus.Caption := 'Y[gIĂB';
    mmSelectedZoomBox : with ZoomBoxNode[SelectedZoomBoxIndex] do
      LabelStatus.Caption := 'IꂽY[g : ' + MapTitlePrefix + IntToStr(TZoomPrm(Data).SerialNo) + '[' + Text + ']';
    mmZoomIn          : LabelStatus.Caption := 'Y[Csc';
    mmZoomOut         : LabelStatus.Caption := 'Y[AEgsc';
    mmJump            : LabelStatus.Caption := 'WvR}hsc';
    mmEditPalette     : LabelStatus.Caption := 'Fpbg̕ύXc';
    mmChangeFont      : LabelStatus.Caption := 'tHgIc';
{$ENDIF}
{$IFDEF LINUX}
    mmNotInitialized  : {do nothing};
    mmInitialize      : LabelStatus.Caption := '档Ԥ ';
    mmNotCommanded    : LabelStatus.Caption := '';
    mmAccessMapFile   : LabelStatus.Caption := 'ޥåץե˥ ';
    mmSetupPrinter    : LabelStatus.Caption := 'ץ ';
    mmPrint           : LabelStatus.Caption := ' ';
    mmEditTitle       : LabelStatus.Caption := 'ȥϤOK򲡤Ƥ';
    mmEditMaxIter     : LabelStatus.Caption := 'ȿ򤷤OK򲡤Ƥ';
    mmDeleteZoomBox   :{LabelStatus.Caption := ''};
    mmShowResizeBox   : LabelStatus.Caption := 'Ȥꤷơ।󥳥ޥɤ¹ԤƤ';
    mmDragResizeBox   :{LabelStatus.Caption := ' '};
    mmShowZoomBox     : LabelStatus.Caption := 'Ȥ򤷤Ƥ';
    mmSelectedZoomBox : with ZoomBoxNode[SelectedZoomBoxIndex] do
      LabelStatus.Caption := '򤵤줿 : ' + MapTitlePrefix + IntToStr(TZoomPrm(Data).SerialNo) + '[' + Text + ']';
    mmZoomIn          : LabelStatus.Caption := '।¹ ';
    mmZoomOut         : LabelStatus.Caption := 'ॢȼ¹ ';
    mmJump            : LabelStatus.Caption := 'ץޥɼ¹ ';
    mmEditPalette     : LabelStatus.Caption := 'ѥåȤѹ ';
    mmChangeFont      : LabelStatus.Caption := 'ե ';
{$ENDIF}
  end;
end;

procedure TFormMain.ChangeScanMode(_Mode: TScanModeList);
begin
  ScanMode := _Mode;
  UpdateMainMenu;
  UpdatePopupMenu;
  UpdateSpeedButton;
end;

procedure TFormMain.CancelCommand;
begin
  case MainMode of
    mmShowResizeBox, mmDragResizeBox : HideResizeBox;
    mmShowZoomBox, mmSelectedZoomBox : HideZoomBox;
  end;
end;

procedure TFormMain.UpdateMainMenu;
begin
  miFileUpdateMapFile.Enabled      := (not Scanning);
  miEditTitle.Enabled              := (MainMode in [mmNotCommanded, mmHelpMode]);
  miEditChangeMaxIteration.Enabled := (MainMode in [mmNotCommanded, mmHelpMode]);
  miEditDeleteZoomBox.Enabled      := (MainMode in [mmSelectedZoomBox, mmHelpMode]);
  miEditPalette.Enabled            := (MainMode in [mmNotCommanded, mmHelpMode]);
  miEditCopyToClipboard.Enabled    := (not Scanning);
  miZoomNewZoomBox.Enabled         := (MainMode in [mmNotCommanded, mmShowZoomBox, mmSelectedZoomBox, mmHelpMode]);
  miZoomDisplayZoomBox.Enabled     := (((MainMode in [mmNotCommanded, mmShowResizeBox, mmDragResizeBox])
                                        and (MapNode(CurrentMap).HasChildren)) or (MainMode in [mmHelpMode]));
  miZoomIn.Enabled                 := (MainMode in [mmShowResizeBox, mmSelectedZoomBox, mmHelpMode]);
  miZoomOut.Enabled                := (((MainMode in [mmNotCommanded]) and (MapNode(WideMap) <> nil))
                                        or (MainMode in [mmHelpMode]));
  miZoomJumpToEntire.Enabled       := (((MainMode in [mmNotCommanded]) and (MapNode(CurrentMap) <> MapNode(EntireMap)))
                                        or (MainMode in [mmHelpMode]));
  miZoomAbort.Enabled              := (MainMode in [mmShowResizeBox, mmDragResizeBox, mmShowZoomBox,
                                        mmSelectedZoomBox, mmHelpMode]);
  miOptionDisplayMapWindow.Enabled := (not Scanning);
  miOptionDisplayMapWindow.Checked := ZoomPrmDlg.Visible;
  miOptionDisplay3DWindow.Enabled  := (not Scanning);
  miOptionDisplay3DWindow.Checked  := View3DDlg.Visible;
  miOptionUseImageFiles.Enabled    := (not Scanning);
  miOptionSelectFont.Enabled       := (not Scanning);
  miHelpMode.Enabled               := (MainMode in [mmNotCommanded, mmHelpMode]);
  miHelpMode.Checked               := (MainMode in [mmHelpMode]);
end;

procedure TFormMain.UpdatePopupMenu;
begin
  piNewZoomBox.Enabled     := miZoomNewZoomBox.Enabled;
  piDisplayZoomBox.Enabled := miZoomDisplayZoomBox.Enabled;
  piZoomIn.Enabled         := miZoomIn.Enabled;
  piZoomOut.Enabled        := miZoomOut.Enabled;
  piJumpToEntire.Enabled   := miZoomJumpToEntire.Enabled;
  piAbort.Caption          := miZoomAbort.Caption;
  piAbort.Enabled          := miZoomAbort.Enabled;
end;

procedure TFormMain.UpdateSpeedButton;
begin
  SpeedButtonUpdateMapFile.Enabled      := miFileUpdateMapFile.Enabled;
  SpeedButtonEditTitle.Enabled          := miEditTitle.Enabled;
  SpeedButtonChangeMaxIteration.Enabled := miEditChangeMaxIteration.Enabled;
  SpeedButtonDeleteZoomBox.Enabled      := miEditDeleteZoomBox.Enabled;
  SpeedButtonEditPalette.Enabled        := miEditPalette.Enabled;
  SpeedButtonCopyToClipboard.Enabled    := miEditCopyToClipboard.Enabled;
  SpeedButtonNewZoomBox.Enabled         := miZoomNewZoomBox.Enabled;
  SpeedButtonDisplayZoomBox.Enabled     := miZoomDisplayZoomBox.Enabled;
  SpeedButtonZoomIn.Enabled             := miZoomIn.Enabled;
  SpeedButtonZoomOut.Enabled            := miZoomOut.Enabled;
  SpeedButtonJumpToEntire.Enabled       := miZoomJumpToEntire.Enabled;
  SpeedButtonAbort.Enabled              := miZoomAbort.Enabled;
  SpeedButtonDisplayMapWindow.Enabled   := miOptionDisplayMapWindow.Enabled;
  SpeedButtonDisplay3DWindow.Enabled    := miOptionDisplay3DWindow.Enabled;
end;

procedure TFormMain.PopupMenuPopup(Sender: TObject);

begin
  UpdatePopupMenu;
end;

procedure TFormMain.miMenuClick(Sender: TObject);
begin
  UpdateMainMenu;
end;

procedure TFormMain.miFileUpdateMapFileClick(Sender: TObject);
begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  ZoomPrmDlg.SaveMapFile(MapFileName);
end;

procedure TFormMain.miFileExitClick(Sender: TObject);
begin
  Close;
end;

procedure TFormMain.miEditTitleClick(Sender: TObject);
begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  if (not NotCommanded) then Exit;
  ChangeMainMode(mmEditTitle);
  EditTitle.Text := MapNode(CurrentMap).Text;
  EditTitle.Visible := True;
  PanelOkCancel.Visible := True;
  EditTitle.SetFocus;
end;

procedure TFormMain.miEditChangeMaxIterationClick(Sender: TObject);
var
  ItemNo, MaxIterIndex: Integer;
begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  if (not NotCommanded) then Exit;
  MaxIterIndex := 0;
  for ItemNo := 0 to MaxIterItems - 1 do begin
    if (MaxIterList[ItemNo] <= TZoomPrm(MapNode(CurrentMap).Data).MaxIter) then
      MaxIterIndex := ItemNo;
  end;
  ChangeMainMode(mmEditMaxIter);
  ListBoxMaxIteration.ItemIndex := MaxIterIndex;
  ListBoxMaxIteration.Visible := True;
  PanelOkCancel.Visible := True;
  ListBoxMaxIteration.SetFocus;
end;

procedure TFormMain.miEditDeleteZoomBoxClick(Sender: TObject);

  procedure DeleteImageFiles(_TreeNode: TTreeNode);
  var
    ChildNode: TTreeNode;
  begin
    DeleteFile(PChar(GetImageFileName(_TreeNode)));
    ChildNode := _TreeNode.GetFirstChild;
    while (ChildNode <> nil) do begin
      DeleteImageFiles(ChildNode);
      ChildNode := _TreeNode.GetNextChild(ChildNode);
    end;
  end;

begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  if (MainMode <> mmSelectedZoomBox) then Exit;
  ChangeMainMode(mmDeleteZoomBox);
{$IFDEF MSWINDOWS}
  if (MessageDlg('̃Y[g폜܂H', mtConfirmation, mbOkCancel, 0) = mrOk) then begin
{$ENDIF}
{$IFDEF LINUX}
  if (MessageDlg('Delete the Zoom Box?', mtConfirmation, mbOkCancel, 0) = mrOk) then begin
{$ENDIF}
    DeleteImageFiles(ZoomBoxNode[SelectedZoomBoxIndex]);
    ZoomPrmDlg.TreeView.Items.BeginUpdate;
    ZoomPrmDlg.TreeView.Items.Delete(ZoomBoxNode[SelectedZoomBoxIndex]);
    ZoomPrmDlg.TreeView.Items.EndUpdate;
    HideZoomBox;
    if (MapNode(CurrentMap).HasChildren) then
      ShowZoomBox
    else
      ChangeMainMode(mmNotCommanded);
  end
  else
    ChangeMainMode(mmSelectedZoomBox);
end;

procedure TFormMain.miEditPaletteClick(Sender: TObject);
var
  SavedRgbL, SavedRgbS1L, SavedRgbS1U, SavedRgbU: TRgbQuad;
  SavedRgbS1Index: Integer;
begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    EditPaletteDlg.Show;
    Exit;
  end;
  ChangeMainMode(mmEditPalette);
  with TZoomPrm(MapNode(CurrentMap).Data) do begin
    SavedRgbL := RgbL;
    SavedRgbS1L := RgbS1L;
    SavedRgbS1U := RgbS1U;
    SavedRgbU := RgbU;
    SavedRgbS1Index := RgbS1Index;
  end;
  if (EditPaletteDlg.ShowModal = mrOk) then
    CurrentMapPaletteChanged
  else begin
    with TZoomPrm(MapNode(CurrentMap).Data) do begin
      RgbL := SavedRgbL;
      RgbS1L := SavedRgbS1L;
      RgbS1U := SavedRgbS1U;
      RgbU := SavedRgbU;
      RgbS1Index := SavedRgbS1Index;
    end;
    ChangeCurrentMapPaletteTemporary;
  end;
  ChangeMainMode(mmNotCommanded);
end;

procedure TFormMain.miEditCopyToClipboardClick(Sender: TObject);
begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  Clipboard.Assign(ImageMap[CurrentMap].Picture);
end;

procedure TFormMain.miZoomNewZoomBoxClick(Sender: TObject);
begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  if (MapNode(CurrentMap).Count >= MaxChildren) then begin
    MessageDlg('Sorry, Zoom Boxes are limited up to ten. Please nest.', mtError, [mbOk], 0);
    Exit;
  end;
  CancelCommand;
  ShowResizeBox;
end;

procedure TFormMain.miZoomDisplayZoomBoxClick(Sender: TObject);
begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  if (not MapNode(CurrentMap).HasChildren) then Exit;
  CancelCommand;
  ShowZoomBox;
end;

procedure TFormMain.miZoomInClick(Sender: TObject);
var
  TmpList: TStrings;
  ItemNo, MaxIterIndex: Integer;
  ZoomInNode: TTreeNode;
  ZoomPrm: TZoomPrm;
begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  case MainMode of
    mmShowResizeBox : begin
      TmpList := TStringList.Create;
      try
        MaxIterIndex := 0;
        for ItemNo := 0 to MaxIterItems - 1 do begin
          TmpList.Add(Format('%5d', [MaxIterList[ItemNo]]));
          if (MaxIterList[ItemNo] <= TZoomPrm(MapNode(CurrentMap).Data).MaxIter) then
            MaxIterIndex := ItemNo;
        end;
        ZoomInDlg.EditTitle.Text := MapTitlePrefix + IntToStr(Succ(ZoomPrmDlg.LastSerial));
        ZoomInDlg.ListBoxMaxIteration.Items := TmpList;
        ZoomInDlg.ListBoxMaxIteration.ItemIndex := MaxIterIndex;
        if (ZoomInDlg.ShowModal = mrOk) then begin
          HideResizeBox;
          with TZoomPrm(MapNode(CurrentMap).Data) do begin
            ZoomPrm := TZoomPrm.Create;
            try
              ZoomPrm.R1 := R1 + (Span * ResizeBox.Left / ImageMap[CurrentMap].Width);
              ZoomPrm.I1 := I1 + (Span * (ImageMap[CurrentMap].Height - (ResizeBox.Top + ResizeBox.Height))
                / ImageMap[CurrentMap].Width);
              ZoomPrm.Span := Span * ResizeBox.Width / ImageMap[CurrentMap].Width;
              ZoomPrm.MaxIter := MaxIterList[ZoomInDlg.ListBoxMaxIteration.ItemIndex];
              ZoomPrm.SerialNo := -1;
              ZoomPrm.RgbL := TZoomPrm(MapNode(CurrentMap).Data).RgbL;
              ZoomPrm.RgbS1L := TZoomPrm(MapNode(CurrentMap).Data).RgbS1L;
              ZoomPrm.RgbS1U := TZoomPrm(MapNode(CurrentMap).Data).RgbS1U;
              ZoomPrm.RgbU := TZoomPrm(MapNode(CurrentMap).Data).RgbU;
              ZoomPrm.RgbS1Index := TZoomPrm(MapNode(CurrentMap).Data).RgbS1Index;
              ZoomInNode := ZoomPrmDlg.AddZoomPrm(MapNode(CurrentMap), ZoomInDlg.EditTitle.Text, ZoomPrm);
            finally
              ZoomPrm.Free;
            end;
            ZoomIn(ZoomInNode);
          end;
        end;
      finally
        TmpList.Free;
      end;
    end;
    mmSelectedZoomBox : begin
      ZoomInNode := ZoomBoxNode[SelectedZoomBoxIndex];
      HideZoomBox;
      ZoomIn(ZoomInNode);
    end;
    else
      Exit;
  end;
  CancelCommand;
end;

procedure TFormMain.miZoomOutClick(Sender: TObject);
begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  CancelCommand;
  ZoomOut;
end;

procedure TFormMain.miZoomJumpToEntireClick(Sender: TObject);
begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  CancelCommand;
  JumpTree(MapNode(EntireMap));
end;

procedure TFormMain.miZoomAbortClick(Sender: TObject);
begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  case MainMode of
    mmShowResizeBox, mmDragResizeBox, mmShowZoomBox, mmSelectedZoomBox :
      CancelCommand;
  end;
end;

procedure TFormMain.miOptionDisplayMapWindowClick(Sender: TObject);

begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    ZoomPrmDlg.Show;
    Exit;
  end;
  if (MainMode = mmNotInitialized) then Exit;
  if (ZoomPrmDlg.Visible) then
    ZoomPrmDlg.Close
  else
    ZoomPrmDlg.Show;
end;

procedure TFormMain.miOptionDisplay3DWindowClick(Sender: TObject);

begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    View3DDlg.Show;
    Exit;
  end;
  if (MainMode = mmNotInitialized) then Exit;
  if (View3DDlg.Visible) then
    View3DDlg.Close
  else
    View3DDlg.Show;
end;

procedure TFormMain.miOptionUseImageFilesClick(Sender: TObject);
begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  if (MainMode = mmNotInitialized) then Exit;
  miOptionUseImageFiles.Checked := not miOptionUseImageFiles.Checked;
end;

procedure TFormMain.miOptionSelectFontClick(Sender: TObject);

begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  FontDialog.Font.Name := FormMain.Font.Name;
  FontDialog.Font.Size := FormMain.Font.Size;
  if (FontDialog.Execute) then
    ChangeFont(FontDialog.Font);
end;

procedure TFormMain.miHelpIntroductionClick(Sender: TObject);

begin
  miHelpMode.Checked := False;
  HelpIntroDlg.Show;
end;

procedure TFormMain.miHelpModeClick(Sender: TObject);
begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.Hide;
    if (EditPaletteDlg.Visible) then
      EditPaletteDlg.Hide;
    ChangeMainMode(mmNotCommanded);
  end
  else begin
    ChangeMainMode(mmHelpMode);
    HelpModeDlg.Show;
    HelpModeDlg.DisplayHelp(Sender);
  end;
end;

procedure TFormMain.miHelpAboutClick(Sender: TObject);

begin
  if (MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    AboutBox.Show;
    Exit;
  end;
  AboutBox.LabelVersion.Caption := 'Ver. ' + VersionString;
  AboutBox.ShowModal;
end;

procedure TFormMain.ButtonOkClick(Sender: TObject);

begin
  case MainMode of
    mmEditTitle : begin
      MapNode(CurrentMap).Text := EditTitle.Text;
      EditTitle.Visible := False;
      UpdateCurrentLabels;
      View3DDlg.Invalidate;
      ChangeMainMode(mmNotCommanded);
    end;
    mmEditMaxIter : begin
      ListBoxMaxIteration.Visible := False;
      if (TZoomPrm(MapNode(CurrentMap).Data).MaxIter <> MaxIterList[ListBoxMaxIteration.ItemIndex]) then begin
        DeleteFile(PChar(GetImageFileName(MapNode(CurrentMap))));
        TZoomPrm(MapNode(CurrentMap).Data).MaxIter := MaxIterList[ListBoxMaxIteration.ItemIndex];
        BeginScan(CurrentMap);
      end
      else
        ButtonCancelClick(Sender);
    end;
  end;
  PanelOkCancel.Visible := False;
end;

procedure TFormMain.ButtonCancelClick(Sender: TObject);

begin
  case MainMode of
    mmEditTitle : begin
      EditTitle.Visible := False;
    end;
    mmEditMaxIter : begin
      ListBoxMaxIteration.Visible := False;
    end;
  end;
  PanelOkCancel.Visible := False;
  ChangeMainMode(mmNotCommanded);
end;

procedure TFormMain.ListBoxMaxIterationKeyPress(Sender: TObject; var Key: Char);

begin
  if (MainMode <> mmEditMaxIter) then Exit;
  case Key of
    #$0d : ButtonOkClick(Sender);
    #$1a : ButtonCancelClick(Sender);
  end;
end;

procedure TFormMain.EditTitleKeyPress(Sender: TObject; var Key: Char);

begin
  if (MainMode <> mmEditTitle) then Exit;
  case Key of
    #$0d : ButtonOkClick(Sender);
    #$1a : ButtonCancelClick(Sender);
  end;
end;

procedure TFormMain.AdjustAspect(var W, H: Integer);

var
  TmpL1, TmpL2: Longint;
begin
  if ((W < ImageMap[CurrentMap].Width div 16) and (H < ImageMap[CurrentMap].Height div 16)) then begin
    W := ImageMap[CurrentMap].Width div 16;
    H := ImageMap[CurrentMap].Height div 16;
  end
  else begin
    TmpL1 := Longint(W) * Longint(ImageMap[CurrentMap].Height);
    TmpL2 := Longint(H) * Longint(ImageMap[CurrentMap].Width);
    if (TmpL1 >= TmpL2) then
      H := TmpL1 div ImageMap[CurrentMap].Width
    else
      W := TmpL2 div ImageMap[CurrentMap].Height;
  end;
end;

function TFormMain.GetAspectH(W: Integer): Integer;
begin
  Result := (Longint(W) * Longint(ImageMap[CurrentMap].Height)) div ImageMap[CurrentMap].Width;
end;

function TFormMain.GetAspectW(H: Integer): Integer;
begin
  Result := (Longint(H) * Longint(ImageMap[CurrentMap].Width)) div ImageMap[CurrentMap].Height;
end;

procedure TFormMain.ResizeResizeBox;
var
  OrgOffset: Integer;
begin
  OrgOffset := ResizeBoxT.Width div 2;
  ResizeBoxT.Left  := ResizeBox.Left + (ResizeBox.Width div 2) - OrgOffset;
  ResizeBoxT.Top   := ResizeBox.Top - OrgOffset;
  ResizeBoxL.Left  := ResizeBox.Left - OrgOffset;
  ResizeBoxL.Top   := ResizeBox.Top + (ResizeBox.Height div 2) - OrgOffset;
  ResizeBoxR.Left  := ResizeBox.Left + ResizeBox.Width - OrgOffset - 1;
  ResizeBoxR.Top   := ResizeBoxL.Top;
  ResizeBoxB.Left  := ResizeBoxT.Left;
  ResizeBoxB.Top   := ResizeBox.Top + ResizeBox.Height - OrgOffset;
  ResizeBoxLT.Left := ResizeBoxL.Left;
  ResizeBoxLT.Top  := ResizeBoxT.Top;
  ResizeBoxRT.Left := ResizeBoxR.Left;
  ResizeBoxRT.Top  := ResizeBoxT.Top;
  ResizeBoxLB.Left := ResizeBoxL.Left;
  ResizeBoxLB.Top  := ResizeBoxB.Top;
  ResizeBoxRB.Left := ResizeBoxR.Left;
  ResizeBoxRB.Top  := ResizeBoxB.Top;
end;

procedure TFormMain.ShowResizeBox;
begin
  with ResizeBox do begin
    Width  := ImageMap[CurrentMap].Width div 2;
    Height := ImageMap[CurrentMap].Height div 2;
    Left   := ImageMap[CurrentMap].Width div 4;
    Top    := ImageMap[CurrentMap].Height div 4;
  end;
  ResizeResizeBox;
  ResizeBox.Show;
  ResizeBoxT.Show;
  ResizeBoxLT.Show;
  ResizeBoxL.Show;
  ResizeBoxLB.Show;
  ResizeBoxRT.Show;
  ResizeBoxR.Show;
  ResizeBoxRB.Show;
  ResizeBoxB.Show;
  ChangeMainMode(mmShowResizeBox);
end;

procedure TFormMain.HideResizeBox;
begin
  ResizeBox.Hide;
  ResizeBoxT.Hide;
  ResizeBoxLT.Hide;
  ResizeBoxL.Hide;
  ResizeBoxLB.Hide;
  ResizeBoxRT.Hide;
  ResizeBoxR.Hide;
  ResizeBoxRB.Hide;
  ResizeBoxB.Hide;
  ChangeMainMode(mmNotCommanded);
end;

procedure TFormMain.ResizeBoxClipT(var TmpW, TmpH: Integer);
begin
  with ResizeBox do
    if ((Top + Height - TmpH) < 3) then begin
      TmpH := Top + Height - 3;
      TmpW := GetAspectW(TmpH);
    end;
end;

procedure TFormMain.ResizeBoxClipL(var TmpW, TmpH: Integer);
begin
  with ResizeBox do
    if ((Left + Width - TmpW) < 3) then begin
      TmpW := Left + Width - 3;
      TmpH := GetAspectH(TmpW);
    end;
end;

procedure TFormMain.ResizeBoxClipR(var TmpW, TmpH: Integer);
begin
  with ResizeBox do
    if ((Left + TmpW) > (ImageMap[CurrentMap].Width - 2)) then begin
      TmpW := ImageMap[CurrentMap].Width - Left;
      TmpH := GetAspectH(TmpW);
    end;
end;

procedure TFormMain.ResizeBoxClipB(var TmpW, TmpH: Integer);
begin
  with ResizeBox do
    if ((Top + TmpH) > (ImageMap[CurrentMap].Height - 2)) then begin
      TmpH := ImageMap[CurrentMap].Height - Top;
      TmpW := GetAspectW(TmpH);
    end;
end;

procedure TFormMain.ResizeBoxMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  DraggingOrgPoint.X := X;
  DraggingOrgPoint.Y := Y;
  ChangeMainMode(mmDragResizeBox);
end;

procedure TFormMain.ResizeBoxMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  ChangeMainMode(mmShowResizeBox);
end;

procedure TFormMain.ResizeBoxMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var
  TmpL, TmpT: Integer;
begin
  if (MainMode <> mmDragResizeBox) then Exit;
  with ResizeBox do begin
    TmpL := Left + (X - DraggingOrgPoint.X);
    TmpT := Top + (Y - DraggingOrgPoint.Y);
    if (TmpL < 3) then
      TmpL := 3;
    if ((TmpL + Width) > (ImageMap[CurrentMap].Width - 2)) then
      TmpL := ImageMap[CurrentMap].Width - Width;
    if (TmpT < 3) then
      TmpT := 3;
    if ((TmpT + Height) > (ImageMap[CurrentMap].Height - 2)) then
      TmpT := ImageMap[CurrentMap].Height - Height;
    Left := TmpL;
    Top := TmpT;
  end;
  ResizeResizeBox;
end;

procedure TFormMain.ResizeBoxLTMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var
  TmpW, TmpH: Integer;
begin
  if (MainMode <> mmDragResizeBox) then Exit;
  with ResizeBox do begin
    TmpW := Width - (X - DraggingOrgPoint.X);
    TmpH := Height - (Y - DraggingOrgPoint.Y);
    AdjustAspect(TmpW, TmpH);
    ResizeBoxClipL(TmpW, TmpH);
    ResizeBoxClipT(TmpW, TmpH);
    Left := Left + Width - TmpW;
    Top := Top + Height - TmpH;
    Width := TmpW;
    Height := TmpH;
  end;
  ResizeResizeBox;
end;

procedure TFormMain.ResizeBoxLBMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var
  TmpW, TmpH: Integer;
begin
  if (MainMode <> mmDragResizeBox) then Exit;
  with ResizeBox do begin
    TmpW := Width - (X - DraggingOrgPoint.X);
    TmpH := Height + (Y - DraggingOrgPoint.Y);
    AdjustAspect(TmpW, TmpH);
    ResizeBoxClipL(TmpW, TmpH);
    ResizeBoxClipB(TmpW, TmpH);
    Left := Left + Width - TmpW;
    Width := TmpW;
    Height := TmpH;
  end;
  ResizeResizeBox;
end;

procedure TFormMain.ResizeBoxRTMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var
  TmpW, TmpH: Integer;
begin
  if (MainMode <> mmDragResizeBox) then Exit;
  with ResizeBox do begin
    TmpW := Width + (X - DraggingOrgPoint.X);
    TmpH := Height - (Y - DraggingOrgPoint.Y);
    AdjustAspect(TmpW, TmpH);
    ResizeBoxClipR(TmpW, TmpH);
    ResizeBoxClipT(TmpW, TmpH);
    Top := Top + Height - TmpH;
    Width := TmpW;
    Height := TmpH;
  end;
  ResizeResizeBox;
end;

procedure TFormMain.ResizeBoxRBMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var
  TmpW, TmpH: Integer;
begin
  if (MainMode <> mmDragResizeBox) then Exit;
  with ResizeBox do begin
    TmpW := Width + (X - DraggingOrgPoint.X);
    TmpH := Height + (Y - DraggingOrgPoint.Y);
    AdjustAspect(TmpW, TmpH);
    ResizeBoxClipR(TmpW, TmpH);
    ResizeBoxClipB(TmpW, TmpH);
    Width := TmpW;
    Height := TmpH;
  end;
  ResizeResizeBox;
end;

procedure TFormMain.ResizeBoxTMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var
  TmpW, TmpH: Integer;
begin
  if (MainMode <> mmDragResizeBox) then Exit;
  with ResizeBox do begin
    TmpW := 0;
    TmpH := Height - (Y - DraggingOrgPoint.Y);
    AdjustAspect(TmpW, TmpH);
    ResizeBoxClipT(TmpW, TmpH);
    TmpW := Width + ((TmpW - Width) div 2);
    ResizeBoxClipL(TmpW, TmpH);
    ResizeBoxClipR(TmpW, TmpH);
    Left := Left + Width - TmpW;
    Width := TmpW + (TmpW - Width);
    TmpH := GetAspectH(Width);
    Top := Top + Height - TmpH;
    Height := TmpH;
  end;
  ResizeResizeBox;
end;

procedure TFormMain.ResizeBoxLMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var
  TmpW, TmpH: Integer;
begin
  if (MainMode <> mmDragResizeBox) then Exit;
  with ResizeBox do begin
    TmpW := Width - (X - DraggingOrgPoint.X);
    TmpH := 0;
    AdjustAspect(TmpW, TmpH);
    ResizeBoxClipL(TmpW, TmpH);
    TmpH := Height + ((TmpH - Height) div 2);
    ResizeBoxClipT(TmpW, TmpH);
    ResizeBoxClipB(TmpW, TmpH);
    Top := Top + Height - TmpH;
    Height := TmpH + (TmpH - Height);
    TmpW := GetAspectW(Height);
    Left := Left + Width - TmpW;
    Width := TmpW;
  end;
  ResizeResizeBox;
end;

procedure TFormMain.ResizeBoxRMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var
  TmpW, TmpH: Integer;
begin
  if (MainMode <> mmDragResizeBox) then Exit;
  with ResizeBox do begin
    TmpW := Width + (X - DraggingOrgPoint.X);
    TmpH := 0;
    AdjustAspect(TmpW, TmpH);
    ResizeBoxClipR(TmpW, TmpH);
    TmpH := Height + ((TmpH - Height) div 2);
    ResizeBoxClipT(TmpW, TmpH);
    ResizeBoxClipB(TmpW, TmpH);
    Top := Top + Height - TmpH;
    Height := TmpH + (TmpH - Height);
    Width := GetAspectW(Height);
  end;
  ResizeResizeBox;
end;

procedure TFormMain.ResizeBoxBMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var
  TmpW, TmpH: Integer;
begin
  if (MainMode <> mmDragResizeBox) then Exit;
  with ResizeBox do begin
    TmpW := 0;
    TmpH := Height + (Y - DraggingOrgPoint.Y);
    AdjustAspect(TmpW, TmpH);
    ResizeBoxClipB(TmpW, TmpH);
    TmpW := Width + ((TmpW - Width) div 2);
    ResizeBoxClipL(TmpW, TmpH);
    ResizeBoxClipR(TmpW, TmpH);
    Left := Left + Width - TmpW;
    Width := TmpW + (TmpW - Width);
    Height := GetAspectH(Width);
  end;
  ResizeResizeBox;
end;

procedure TFormMain.GetZoomBoxRect(_Client, _Child: TTreeNode; _Image: TImage; var _ResultRect: TRect);
var
  Scale: TFLoat;
begin
  with ZoomPrmDlg.TreeView do begin
    Scale := TZoomPrm(_Child.Data).Span / TZoomPrm(_Client.Data).Span;
    _ResultRect.Left := Round(_Image.Width * (TZoomPrm(_Child.Data).R1 - TZoomPrm(_Client.Data).R1)
      / TZoomPrm(_Client.Data).Span);
    _ResultRect.Top := _Image.Height - Round((_Image.Height * Scale)
      + (_Image.Height * (TZoomPrm(_Child.Data).I1 - TZoomPrm(_Client.Data).I1)
        / (TZoomPrm(_Client.Data).Span * _Image.Height / _Image.Width)));
    _ResultRect.Right := _ResultRect.Left + Round(_Image.Width * Scale);
    _ResultRect.Bottom := _ResultRect.Top + Round(_Image.Height * Scale);
  end;
end;

procedure TFormMain.ShowZoomBox;
var
  TmpI: Integer;
  ZoomBoxRect: TRect;
  ChildNode: TTreeNode;
begin
  SelectedZoomBoxIndex := 0;
  TmpI := 1;
  ChildNode := MapNode(CurrentMap).GetFirstChild;
  while (TmpI <= MaxChildren) do begin
    ZoomBoxNode[TmpI] := ChildNode;
    ZoomBox[TmpI].Tag := TmpI;
    if (ChildNode <> nil) then begin
      GetZoomBoxRect(MapNode(CurrentMap), ChildNode, ImageMap[CurrentMap], ZoomBoxRect);
      with ZoomBoxRect do begin
        ZoomBox[TmpI].Left := Left + 1(**);
        ZoomBox[TmpI].Top := Top;
        ZoomBox[TmpI].Width := Right - Left;
        ZoomBox[TmpI].Height := Bottom - Top;
      end;
      ZoomBox[TmpI].Pen.Color := clWhite;
      ZoomBox[TmpI].Show;
    end;
    inc(TmpI);
    ChildNode := MapNode(CurrentMap).GetNextChild(ChildNode);
  end;
  ChangeMainMode(mmShowZoomBox);
end;

procedure TFormMain.HideZoomBox;
var
  TmpI: Integer;
begin
  for TmpI := 1 to MaxChildren do begin
    ZoomBox[TmpI].Hide;
    ZoomBox[TmpI].Tag := 0;
    ZoomBoxNode[TmpI] := nil;
  end;
  SelectedZoomBoxIndex := 0;
  ChangeMainMode(mmNotCommanded);
end;

procedure TFormMain.ZoomBoxMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  TmpI: Integer;
begin
  if (not (MainMode in [mmShowZoomBox, mmSelectedZoomBox])) then Exit;
  for TmpI := 1 to MaxChildren do
    ZoomBox[TmpI].Pen.Color := clWhite;
  (Sender as TShape).Pen.Color := clRed;
  SelectedZoomBoxIndex := (Sender as TShape).Tag;
  ChangeMainMode(mmSelectedZoomBox);
end;

procedure TFormMain.ImageCurrentClick(Sender: TObject);
begin
  if (MainMode <> mmSelectedZoomBox) then Exit;
  HideZoomBox;
  ShowZoomBox;
end;

procedure TFormMain.ShowGuideEntire;
var
  ZoomBoxRect: TRect;
begin
  if (MapNode(CurrentMap).Level = 0) then begin
    HideGuideEntire;
    Exit;
  end;
  GetZoomBoxRect(MapNode(EntireMap), MapNode(CurrentMap), ImageMap[EntireMap], ZoomBoxRect);
  with ZoomBoxRect do begin
    EntireGuideLineH.Top := Top + ((Bottom - Top) div 2);
    EntireGuideLineV.Left := Left + ((Right - Left) div 2);
    EntireGuideLineH.Left := EntireGuideLineV.Left - (EntireGuideLineH.Width div 2);
    EntireGuideLineV.Top := EntireGuideLineH.Top - (EntireGuideLineV.Height div 2);
  end;
  EntireGuideLineH.Show;
  EntireGuideLineV.Show;
end;

procedure TFormMain.HideGuideEntire;
begin
  EntireGuideLineH.Hide;
  EntireGuideLineV.Hide;
end;

procedure TFormMain.ShowGuideWide;
var
  ZoomBoxRect: TRect;
begin
  if (MapNode(WideMap) = nil) then Exit;
  GetZoomBoxRect(MapNode(WideMap), MapNode(CurrentMap), ImageMap[WideMap], ZoomBoxRect);
  with ZoomBoxRect do begin
    WideGuideBox.Left := Left;
    WideGuideBox.Top := Top;
    WideGuideBox.Width := Right - Left;
    WideGuideBox.Height := Bottom - Top;
  end;
  WideGuideBox.Show;
end;

procedure TFormMain.HideGuideWide;
begin
  WideGuideBox.Hide;
end;

procedure TFormMain.ClickToHelp(Sender: TObject);
begin
  if (MainMode <> mmHelpMode) then Exit;
  HelpModeDlg.DisplayHelp(Sender);
end;

end.

