unit MePalEdt;

(*------------------------------------------------------------------------------
    Mandelbrot Set Explorer
    Copyright (C) 2003 Chiaki Nakajima

    This file is part of Mandelbrot Set Explorer [MSE].

    MSE is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    MSE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with MSE; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
------------------------------------------------------------------------------*)

interface

uses
{$IFDEF MSWINDOWS}
  Graphics, Controls, Forms, Dialogs, StdCtrls, ExtCtrls, ComCtrls, Buttons,
{$ENDIF}
{$IFDEF LINUX}
  QTypes, QGraphics, QControls, QForms, QDialogs, QStdCtrls, QExtCtrls, QComCtrls, QButtons,
{$ENDIF}
  SysUtils, Types, Classes, Variants, MeBmp, MeTree;

const
  SectionEditPalette = 'EditPalette';

type
  TSpeedButtonList = (sbL, sbS1L, sbS1U, sbU);
  TEditPaletteDlg = class(TForm)
    ButtonOk: TButton;
    ButtonCancel: TButton;
    PanelColorSample: TPanel;
    TrackBarSeparator1: TTrackBar;
    ImageColorSample: TImage;
    SpeedButtonL: TSpeedButton;
    SpeedButtonS1L: TSpeedButton;
    SpeedButtonS1U: TSpeedButton;
    SpeedButtonU: TSpeedButton;
    PanelPalette: TPanel;
    ImagePalette: TImage;
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormHide(Sender: TObject);
    procedure LoadIniFile;
    procedure SaveIniFile;
    procedure MakeColorSample(var _Image: TImage);
    procedure UpdateImagePalette;
    procedure UpdateButtonColor(var _SpeedButton: TSpeedButton);
    procedure TrackBarSeparator1Change(Sender: TObject);
    function  SelectedSpeedButton: TSpeedButton;
    procedure ImageColorSampleMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure ImageColorSampleMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure ClickToHelp(Sender: TObject);
  private
    { Private }
  public
    { Public }
  end;

var
  EditPaletteDlg: TEditPaletteDlg;

implementation

uses
  MeIni, MeHelp, MeMain;

{$IFDEF MSWINDOWS}
{$R *.dfm}
{$ENDIF}
{$IFDEF LINUX}
{$R *.xfm}
{$ENDIF}

procedure TEditPaletteDlg.FormCreate(Sender: TObject);
var
  Bmp: TBitmap;
begin
  Bmp := nil;
  try
    Bmp := TBitmap.Create;
    Bmp.PixelFormat := pf32bit;
    Bmp.Width := MeMain.PalMax - 2;
    Bmp.Height := ImagePalette.Height;
    ImagePalette.Picture.Graphic := Bmp;
    Bmp.Width := ImageColorSample.Width;
    Bmp.Height := ImageColorSample.Height;
    ImageColorSample.Picture.Graphic := Bmp;
    Bmp.Width := SpeedButtonL.Width - 8;
    Bmp.Height := SpeedButtonL.Height - 8;
    SpeedButtonL.Glyph := Bmp;
    SpeedButtonS1L.Glyph := Bmp;
    SpeedButtonS1U.Glyph := Bmp;
    SpeedButtonU.Glyph := Bmp;
  finally
    Bmp.Free;
  end;
  SpeedButtonL.Glyph.Canvas.Brush.Style := bsSolid;
  SpeedButtonS1L.Glyph.Canvas.Brush.Style := bsSolid;
  SpeedButtonS1U.Glyph.Canvas.Brush.Style := bsSolid;
  SpeedButtonU.Glyph.Canvas.Brush.Style := bsSolid;
  SpeedButtonL.Tag := Ord(sbL);
  SpeedButtonS1L.Tag := Ord(sbS1L);
  SpeedButtonS1U.Tag := Ord(sbS1U);
  SpeedButtonU.Tag := Ord(sbU);
end;

procedure TEditPaletteDlg.FormShow(Sender: TObject);
begin
  LoadIniFile;
  TrackBarSeparator1.Position := TZoomPrm(FormMain.MapNode(CurrentMap).Data).RgbS1Index;
  UpdateImagePalette;
  MakeColorSample(ImageColorSample);
  UpdateButtonColor(SpeedButtonL);
  UpdateButtonColor(SpeedButtonS1L);
  UpdateButtonColor(SpeedButtonS1U);
  UpdateButtonColor(SpeedButtonU);
  if (FormMain.MainMode = mmHelpMode) then
    HelpModeDlg.DisplayHelp(FormMain.miEditPalette);
end;

procedure TEditPaletteDlg.FormHide(Sender: TObject);
begin
  SaveIniFile;
end;

procedure TEditPaletteDlg.LoadIniFile;
begin
  Left := Ini.ReadInteger(SectionEditPalette, MeMain.StringLeft, 0);
  Top := Ini.ReadInteger(SectionEditPalette, MeMain.StringTop, 0);
end;

procedure TEditPaletteDlg.SaveIniFile;
begin
  Ini.WriteInteger(SectionEditPalette, MeMain.StringLeft, Left);
  Ini.WriteInteger(SectionEditPalette, MeMain.StringTop, Top);
end;

procedure TEditPaletteDlg.UpdateImagePalette;
var
  Px, Py: Integer;
  Rgb: TRgbQuad;
begin
  ImagePalette.Picture.Bitmap.Canvas.Lock;
  with FormMain do begin
    for Px := 1 to MeMain.PalMax - 2 do begin
      Rgb := ImageBmp256[CurrentMap].Palette^[Px];
      for Py := 0 to ImagePalette.Height - 1 do
        PRgbQuadArray(ImagePalette.Picture.Bitmap.ScanLine[Py])^[Px - 1] := Rgb;
    end;
  end;
  ImagePalette.Picture.Bitmap.Canvas.UnLock;
  ImagePalette.Invalidate;
end;

procedure TEditPaletteDlg.UpdateButtonColor(var _SpeedButton: TSpeedButton);
var
  Col: TColor;
begin
  with _SpeedButton do begin
    Glyph.Canvas.Lock;
    Col := clBlack;
    case TSpeedButtonList(Tag) of
      sbL : Col := RgbToColor(TZoomPrm(FormMain.MapNode(CurrentMap).Data).RgbL);
      sbS1L : Col := RgbToColor(TZoomPrm(FormMain.MapNode(CurrentMap).Data).RgbS1L);
      sbS1U : Col := RgbToColor(TZoomPrm(FormMain.MapNode(CurrentMap).Data).RgbS1U);
      sbU : Col := RgbToColor(TZoomPrm(FormMain.MapNode(CurrentMap).Data).RgbU);
    end;
    Glyph.Canvas.Brush.Color := Col;
    Glyph.Canvas.FillRect(Rect(0, 0, Glyph.Width, Glyph.Height));
    Glyph.Canvas.Pen.Color := clBlack;
    Glyph.Canvas.Polyline([Point(0, 0), Point(Glyph.Width - 1, 0),
      Point(Glyph.Width - 1, Glyph.Height - 1), Point(0, Glyph.Height - 1), Point(0, 0)]);
    Glyph.Canvas.Pixels[0, Glyph.Height - 1] := Col xor $00808080;
    Glyph.Canvas.Unlock;
    Invalidate;
  end;
end;

procedure TEditPaletteDlg.MakeColorSample(var _Image: TImage);
var
  Px, Py, OrgY, W, H: Integer;
  NormX, NormY, R0, G0, B0: Single;

  function Clip(_A: Single): Single;
  begin
    if (_A < 0.0) then
      Result := 0.0
    else if (_A > 1.0) then
      Result := 1.0
    else
      Result := _A;
  end;

  procedure SetRgb(_X, _Y: Integer; _NormR, _NormG, _NormB: Single);
  var
    Rgb: TRgbQuad;
  begin
    Rgb.rgbBlue := Round(255.0 * _NormB);
    Rgb.rgbGreen := Round(255.0 * _NormG);
    Rgb.rgbRed := Round(255.0 * _NormR);
    Rgb.rgbReserved := $00;
    PRgbQuadArray(_Image.Picture.Bitmap.ScanLine[_Y])^[_X] := Rgb;
  end;

begin
  _Image.Picture.Bitmap.Canvas.Lock;
  W := _Image.Picture.Bitmap.Width;
  H := _Image.Picture.Bitmap.Height;
  OrgY := H div 2;
  for Px := 0 to W - 1 do begin
    NormX := Px / (W - 1);
    R0 := Clip(Abs(- 6.0 * NormX + 3.0) - 1.0);
    G0 := Clip(- Abs(6.0 * NormX - 2.0) + 2.0);
    B0 := Clip(- Abs(6.0 * NormX - 4.0) + 2.0);
    SetRgb(Px, OrgY, R0, G0, B0);
    for Py := 1 to (H - 1) - OrgY do begin
      NormY := Py / ((H - 1) - OrgY);
      SetRgb(Px, (H - 1) - OrgY - Py, NormY + (1.0 - NormY) * R0,
        NormY + (1.0 - NormY) * G0, NormY + (1.0 - NormY) * B0);
    end;
    for Py := - (OrgY - 1) to - 1 do begin
      NormY := Py / (OrgY - 1);
      SetRgb(Px, OrgY - Py, (1.0 + NormY) * R0, (1.0 + NormY) * G0, (1.0 + NormY) * B0);
    end;
  end;
  _Image.Picture.Bitmap.Canvas.Unlock;
  _Image.Invalidate;
end;

function TEditPaletteDlg.SelectedSpeedButton: TSpeedButton;
begin
  Result := nil;
  if (SpeedButtonL.Down) then
    Result := SpeedButtonL
  else if (SpeedButtonS1L.Down) then
    Result := SpeedButtonS1L
  else if (SpeedButtonS1U.Down) then
    Result := SpeedButtonS1U
  else if (SpeedButtonU.Down) then
    Result := SpeedButtonU;
end;

procedure TEditPaletteDlg.TrackBarSeparator1Change(Sender: TObject);
begin
  if (FormMain.MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  if (not EditPaletteDlg.Visible) then Exit;
  TZoomPrm(FormMain.MapNode(CurrentMap).Data).RgbS1Index:= TrackBarSeparator1.Position;
  FormMain.ChangeCurrentMapPaletteTemporary;
  UpdateImagePalette;
end;

procedure TEditPaletteDlg.ImageColorSampleMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if (FormMain.MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  if (not (mbLeft = Button)) then Exit;
  ImageColorSampleMouseMove(Sender, [ssLeft], X, Y);
end;

procedure TEditPaletteDlg.ImageColorSampleMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
var
  TmpSb: TSpeedButton;
  Rgb: TRgbQuad;
begin
  if (FormMain.MainMode = mmHelpMode) then begin
    Exit;
  end;
  if (not ((ssLeft in Shift) and (X in [0..ImageColorSample.Width - 1]) and (Y in [0..ImageColorSample.Height - 1]))) then Exit;
  TmpSb := SelectedSpeedButton;
  Rgb := PRgbQuadArray(ImageColorSample.Picture.Bitmap.ScanLine[Y])^[X];
  case TSpeedButtonList(TmpSb.Tag) of
    sbL : TZoomPrm(FormMain.MapNode(CurrentMap).Data).RgbL := Rgb;
    sbS1L : TZoomPrm(FormMain.MapNode(CurrentMap).Data).RgbS1L := Rgb;
    sbS1U : TZoomPrm(FormMain.MapNode(CurrentMap).Data).RgbS1U := Rgb;
    sbU : TZoomPrm(FormMain.MapNode(CurrentMap).Data).RgbU := Rgb;
  end;
  FormMain.ChangeCurrentMapPaletteTemporary;
  UpdateButtonColor(TmpSb);
  UpdateImagePalette;
end;

procedure TEditPaletteDlg.ClickToHelp(Sender: TObject);
begin
  if (FormMain.MainMode = mmHelpMode) then
    HelpModeDlg.DisplayHelp(Sender);
end;

end.
