unit MeTree;

(*------------------------------------------------------------------------------
    Mandelbrot Set Explorer
    Copyright (C) 2003 Chiaki Nakajima

    This file is part of Mandelbrot Set Explorer [MSE].

    MSE is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    MSE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with MSE; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
------------------------------------------------------------------------------*)

interface

uses
{$IFDEF MSWINDOWS}
  Graphics, Controls, Forms, Dialogs, StdCtrls, ComCtrls, ExtCtrls, Menus,
{$ENDIF}
{$IFDEF LINUX}
  QTypes, QGraphics, QControls, QForms, QDialogs, QStdCtrls, QComCtrls, QExtCtrls, QMenus,
{$ENDIF}
  SysUtils, Types, Classes, Variants, MeBmp, MeMain;

const
  StringComment = '//';
  MapTreeSection = '[MapTree]';
  EndSection = '[EndSection]';
  SectionTree  = 'Tree';
  StringParent = 'GotoParent';
  StringChild  = 'GotoChild';

type
  TZoomPrm = class(TObject)
    R1, I1, Span: TFloat;
    MaxIter: Integer;
    SerialNo: Longint;
    RgbS1Index: Integer;
    RgbL, RgbS1L, RgbS1U, RgbU: TRgbQuad;
  end;
  TZoomPrmDlg = class(TForm)
    PanelTreeView: TPanel;
    TreeView: TTreeView;
    PopupMenu: TPopupMenu;
    piJump: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure LoadIniFile;
    procedure SaveIniFile;
    procedure LoadMapFile(_FileName: String);
    procedure SaveMapFile(_FileName: String);
    function  AddZoomPrm(_TreeNode: TTreeNode; _Title: String; _ZoomPrm: TZoomPrm): TTreeNode;
    procedure FreeZoomPrm(_TreeNode: TTreeNode);
    procedure piJumpClick(Sender: TObject);
    procedure UpdatePopupMenu(Sender: TObject);
{$IFDEF MSWINDOWS}
    procedure TreeViewEdited(Sender: TObject; Node: TTreeNode; var S: String);
{$ENDIF}
{$IFDEF LINUX}
    procedure TreeViewEdited(Sender: TObject; Node: TTreeNode; var S: WideString);
{$ENDIF}
    procedure ClickToHelp(Sender: TObject);
  private
    { Private }
  public
    { Public }
    LastSerial: Longint;
  end;

var
  ZoomPrmDlg: TZoomPrmDlg;

implementation

uses
  MeView3D, MeIni, MeHelp;

{$IFDEF MSWINDOWS}
{$R *.dfm}
{$ENDIF}
{$IFDEF LINUX}
{$R *.xfm}
{$ENDIF}

procedure TZoomPrmDlg.FormCreate(Sender: TObject);
begin
  LastSerial := 0;
  ZoomPrmDlg.WindowState := wsMinimized;
end;

procedure TZoomPrmDlg.LoadIniFile;

var
  TmpLeft, TmpTop, TmpWidth, TmpHeight: Integer;
begin
  TmpLeft := Ini.ReadInteger(SectionTree, MeMain.StringLeft, 0);
  TmpTop := Ini.ReadInteger(SectionTree, MeMain.StringTop, 0);
  TmpWidth := Ini.ReadInteger(SectionTree, MeMain.StringWidth, 300);
  TmpHeight := Ini.ReadInteger(SectionTree, MeMain.StringHeight, 400);
  SetBounds(TmpLeft, TmpTop, TmpWidth, TmpHeight);
end;

procedure TZoomPrmDlg.SaveIniFile;
begin
  Ini.WriteInteger(SectionTree, MeMain.StringLeft, Left);
  Ini.WriteInteger(SectionTree, MeMain.StringTop, Top);
  Ini.WriteInteger(SectionTree, MeMain.StringWidth, Width);
  Ini.WriteInteger(SectionTree, MeMain.StringHeight, Height);
end;

procedure TZoomPrmDlg.LoadMapFile(_FileName: String);
var
  CurrentFile: TextFile;
  TmpS: String;

  function GetLine: String;
  var
    TmpS: String;
  begin
    TmpS := '';
    repeat
      Readln(CurrentFile, TmpS);
    until (Eof(CurrentFile) or ((TmpS <> '') and (Pos(StringComment, TmpS) <> 1)));
    GetLine := TmpS;
  end;

  procedure ReadItem(_ParentNode: TTreeNode);
  var
    ZoomPrm: TZoomPrm;
    TmpS: String;
    Node: TTreeNode;
    Code: Integer;

    procedure ReadRgb(var _Rgb: TRgbQuad);
    begin
      with _Rgb do begin
        Val(GetLine, rgbReserved, Code);
        Val(GetLine, rgbRed, Code);
        Val(GetLine, rgbGreen, Code);
        Val(GetLine, rgbBlue, Code);
      end;
    end;

  begin
    ZoomPrm := TZoomPrm.Create;
    try
      with ZoomPrm do begin
        Code := 0;
        TmpS := GetLine;
        Val(GetLine, R1, Code);
        Val(GetLine, I1, Code);
        Val(GetLine, Span, Code);
        Val(GetLine, MaxIter, Code);
        Val(GetLine, SerialNo, Code);
        ReadRgb(RgbL);
        ReadRgb(RgbS1L);
        ReadRgb(RgbS1U);
        ReadRgb(RgbU);
        Val(GetLine, RgbS1Index, Code);
        Node:= AddZoomPrm(_ParentNode, TmpS, ZoomPrm);
        repeat
          TmpS := GetLine;
          if (Pos(StringChild, TmpS) = 1) then
            ReadItem(Node);
        until (Eof(CurrentFile) or (Pos(StringParent, TmpS) = 1));
      end;
    finally
      ZoomPrm.Free;
    end;
  end;

begin
  if (not FileExists(_FileName)) then Exit;
  AssignFile(CurrentFile, _FileName);
  Reset(CurrentFile);
  while (not Eof(CurrentFile)) do begin
    Readln(CurrentFile, TmpS);
    if ((TmpS <> '') and (Pos(StringComment, TmpS) <> 1)) then begin
      if (Pos(MapTreeSection, TmpS) = 1) then begin
        if (TreeView.Items.Count > 0) then begin
          TreeView.Items.BeginUpdate;
          TreeView.Items.Delete(TreeView.Items[0]);
          TreeView.Items.EndUpdate;
        end;
        if (not Eof(CurrentFile)) then
          ReadItem(nil);
        while ((not Eof(CurrentFile)) and (Pos(EndSection, GetLine) <> 1)) do {Nothing};
        TreeView.FullExpand;
      end;
    end;
  end;
  CloseFile(CurrentFile);
end;

procedure TZoomPrmDlg.SaveMapFile(_FileName: String);
var
  CurrentFile: TextFile;

  procedure WriteItem(_Node: TTreeNode);
  var
    ChildNode: TTreeNode;

    procedure WriteRgb(var _Rgb: TRgbQuad);
    begin
      with _Rgb do begin
        Writeln(CurrentFile, rgbReserved);
        Writeln(CurrentFile, rgbRed);
        Writeln(CurrentFile, rgbGreen);
        Writeln(CurrentFile, rgbBlue);
      end;
    end;

  begin
    with _Node do begin
      Writeln(CurrentFile, Text);
      Writeln(CurrentFile, TZoomPrm(Data).R1);
      Writeln(CurrentFile, TZoomPrm(Data).I1);
      Writeln(CurrentFile, TZoomPrm(Data).Span);
      Writeln(CurrentFile, TZoomPrm(Data).MaxIter);
      Writeln(CurrentFile, TZoomPrm(Data).SerialNo);
      WriteRgb(TZoomPrm(Data).RgbL);
      WriteRgb(TZoomPrm(Data).RgbS1L);
      WriteRgb(TZoomPrm(Data).RgbS1U);
      WriteRgb(TZoomPrm(Data).RgbU);
      Writeln(CurrentFile, TZoomPrm(Data).RgbS1Index);
      ChildNode := GetFirstChild;
      while (ChildNode <> nil) do begin
        Writeln(CurrentFile);
        Writeln(CurrentFile, StringChild);
        WriteItem(ChildNode);
        ChildNode := GetNextChild(ChildNode);
      end;
      if (_Node.Parent <> nil) then
        Writeln(CurrentFile, StringParent);
    end;
  end;

begin
  if (_FileName = '') then Exit;
  AssignFile(CurrentFile, _FileName);
  Rewrite(CurrentFile);
  Writeln(CurrentFile, StringComment);
  Writeln(CurrentFile, StringComment, '  Mandelbrot Tour : Map File');
  Writeln(CurrentFile, StringComment, '  FileName : [', _FileName, ']');
  Writeln(CurrentFile, StringComment);
  Writeln(CurrentFile);
  Writeln(CurrentFile, MapTreeSection);
  WriteItem(TreeView.Items[0]);
  Writeln(CurrentFile, EndSection);
  Writeln(CurrentFile);
  Writeln(CurrentFile, StringComment, '  End of File.');
  CloseFile(CurrentFile);
end;

function TZoomPrmDlg.AddZoomPrm(_TreeNode: TTreeNode; _Title: String; _ZoomPrm: TZoomPrm): TTreeNode;
var
  ZoomPrm: TZoomPrm;
begin
  ZoomPrm := TZoomPrm.Create;
  with ZoomPrm do begin
    R1 := _ZoomPrm.R1;
    I1 := _ZoomPrm.I1;
    Span := _ZoomPrm.Span;
    MaxIter := _ZoomPrm.MaxIter;
    if (_ZoomPrm.SerialNo < 0) then begin
      Inc(LastSerial);
      SerialNo := LastSerial;
    end
    else begin
      if (LastSerial < _ZoomPrm.SerialNo) then
        LastSerial := _ZoomPrm.SerialNo;
      SerialNo := _ZoomPrm.SerialNo;
    end;
    RgbL := _ZoomPrm.RgbL;
    RgbS1L := _ZoomPrm.RgbS1L;
    RgbS1U := _ZoomPrm.RgbS1U;
    RgbU := _ZoomPrm.RgbU;
    RgbS1Index := _ZoomPrm.RgbS1Index;
  end;
  AddZoomPrm := TreeView.Items.AddChildObject(_TreeNode, _Title, ZoomPrm);
  if (_TreeNode <> nil) then
    _TreeNode.Expand(False);
end;

procedure TZoomPrmDlg.FreeZoomPrm(_TreeNode: TTreeNode);
var
  ChildNode: TTreeNode;
begin
  TZoomPrm(_TreeNode.Data).Free;
  ChildNode := _TreeNode.GetFirstChild;
  while (ChildNode <> nil) do begin
    FreeZoomPrm(ChildNode);
    ChildNode := _TreeNode.GetNextChild(ChildNode);
  end;
 {_TreeNode.Delete;}
end;

procedure TZoomPrmDlg.UpdatePopupMenu(Sender: TObject);
begin
  piJump.Enabled := (FormMain.MainMode = mmNotCommanded);
end;

procedure TZoomPrmDlg.piJumpClick(Sender: TObject);
begin
  if (FormMain.MainMode <> mmNotCommanded) then Exit;
  FormMain.JumpTree(TreeView.Selected);
end;

{$IFDEF MSWINDOWS}
procedure TZoomPrmDlg.TreeViewEdited(Sender: TObject; Node: TTreeNode; var S: String);
{$ENDIF}
{$IFDEF LINUX}
procedure TZoomPrmDlg.TreeViewEdited(Sender: TObject; Node: TTreeNode; var S: WideString);
{$ENDIF}
begin
  if (Node = FormMain.MapNode(CurrentMap)) then begin
    Node.Text := S;
    FormMain.UpdateCurrentLabels
  end;
end;

procedure TZoomPrmDlg.ClickToHelp(Sender: TObject);
begin
  if (FormMain.MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
end;

end.
