unit MeView3D;

(*------------------------------------------------------------------------------
    Mandelbrot Set Explorer
    Copyright (C) 2003 Chiaki Nakajima

    This file is part of Mandelbrot Set Explorer [MSE].

    MSE is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    MSE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with MSE; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
------------------------------------------------------------------------------*)

interface

uses
{$IFDEF MSWINDOWS}
  Graphics, Controls, Forms, Dialogs, StdCtrls, ExtCtrls, Buttons,
{$ENDIF}
{$IFDEF LINUX}
  QTypes, QGraphics, QControls, QForms, QDialogs, QStdCtrls, QExtCtrls, QButtons,
{$ENDIF}
  SysUtils, Types, Classes, Variants, MeMain, MeBmp;

const
  SectionView3D = 'View3D';

type
  RotDirList = (RotN, RotU, RotD, RotL, RotR);
  DepthDirList = (DepthU, DepthD);
  TView3DDlg = class(TForm)
    PanelClient: TPanel;
    PanelBottom: TPanel;
    PanelSwitches: TPanel;
    SpeedButtonRotL: TSpeedButton;
    SpeedButtonRotU: TSpeedButton;
    SpeedButtonRotD: TSpeedButton;
    SpeedButtonRotR: TSpeedButton;
    SpeedButtonDepthU: TSpeedButton;
    SpeedButtonDepthD: TSpeedButton;
    Panel3DBackground: TPanel;
    Image3D: TImage;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure LoadIniFile;
    procedure SaveIniFile;
    function  Rad(_Deg: Single): Single;
    function  Deg(_Rad: Single): Single;
    function  RotThetaX(_Dir: RotDirList): Integer;
    function  RotThetaZ(_Dir: RotDirList): Integer;
    procedure MakeRotBmp;
    procedure DrawRotBmp;
    procedure SrcBmpChanged;
    procedure SpeedButtonRotClick(Sender: TObject);
    procedure SpeedButtonDepthClick(Sender: TObject);
    procedure ClickToHelp(Sender: TObject);
  private
    { Private }
    MaxHeight: Integer;
  public
    { Public }
    SrcBmp: TBmp256;
    RotDir: RotDirList;
    DepthDir: DepthDirList;
    ThetaX: Integer;
    ThetaZ: Integer;
    ThetaStep: Integer;
    FormWidthPrev: Integer;
    FormHeightPrev: Integer;
  end;

var
  View3DDlg: TView3DDlg;

implementation

uses
  MeIni, MeHelp;

{$IFDEF MSWINDOWS}
{$R *.dfm}
{$ENDIF}
{$IFDEF LINUX}
{$R *.xfm}
{$ENDIF}

procedure TView3DDlg.FormCreate(Sender: TObject);
var
  Bmp: TBitmap;
  TmpI: Integer;
begin
  MaxHeight := MeMain.PalMax div 2;
  SrcBmp := TBmp256.Create;
  TmpI := (((Round(Sqrt(Sqr(Longint(MeMain.ImageBmpWidth))
    + Sqr(Longint(MeMain.ImageBmpHeight)))) + 31) div 32) + 1) * 32;
  Bmp := nil;
  try
    Bmp := TBitmap.Create;
    Bmp.PixelFormat := pf32bit;
    Bmp.Width := TmpI;
    Bmp.Height := TmpI;
    Image3D.Picture.Graphic := Bmp;
  finally
    Bmp.Free;
  end;
  RotDir := RotN;
  DepthDir := DepthU;
  ThetaX := 30;
  ThetaZ := 30;
  ThetaStep := 10;
  FormWidthPrev := Width;
  FormHeightPrev := Height;
  View3DDlg.WindowState := wsMinimized;
end;

procedure TView3DDlg.FormDestroy(Sender: TObject);
begin
  SrcBmp.Destroy;
end;

procedure TView3DDlg.LoadIniFile;
var
  TmpLeft, TmpTop, TmpWidth, TmpHeight: Integer;
begin
  TmpLeft := Ini.ReadInteger(SectionView3D, MeMain.StringLeft, 0);
  TmpTop := Ini.ReadInteger(SectionView3D, MeMain.StringTop, 0);
  TmpWidth := Ini.ReadInteger(SectionView3D, MeMain.StringWidth, 400);
  TmpHeight := Ini.ReadInteger(SectionView3D, MeMain.StringHeight, 503);
  SetBounds(TmpLeft, TmpTop, TmpWidth, TmpHeight);
end;

procedure TView3DDlg.SaveIniFile;
begin
  Ini.WriteInteger(SectionView3D, MeMain.StringLeft, Left);
  Ini.WriteInteger(SectionView3D, MeMain.StringTop, Top);
  Ini.WriteInteger(SectionView3D, MeMain.StringWidth, Width);
  Ini.WriteInteger(SectionView3D, MeMain.StringHeight, Height);
end;

function TView3DDlg.Rad(_Deg: Single): Single;
begin
  Result := _Deg * PI / 180.0;
end;

function TView3DDlg.Deg(_Rad: Single): Single;
begin
  Result := _Rad * 180.0 / PI;
end;

procedure TView3DDlg.MakeRotBmp;
var
  Cx, Cy, Cz, dCx, dCy, oCx, oCy, OfsCx, OfsCy, sC0x, sC0y, Sx, dTopSy, dBottomSy, BottomSy, MaxSz,
  SrcBmpWidth, SrcBmpHeight, RotBmpWidth, RotBmpHeight, PalMax_1, TmpI: Integer;
  sCx0x, sCx0y, SinX, CosX, SinZ, CosZ: Single;
  sCy0x, sCy0y: array [0..MeMain.ImageBmpWidth - 1] of Single;
  IdxCz, IdxSz: array [0..MeMain.PalMax - 1] of Integer;
  DrawnSy: array [0..MeMain.ImageBmpWidth + MeMain.ImageBmpHeight] of Integer;
  SrcBmpX, SrcBmpY, RotBmpX, RotBmpY: Integer;
  pDstLine: PRgbQuadArray;
  RgbQ: TRgbQuad;

  function MulDiv(_A, _B, _C: Integer): Integer;
  begin
    Result := (_A * _B) div _C;
  end;

begin
  Image3D.Picture.Bitmap.Canvas.Lock;
  SrcBmpWidth := MeMain.ImageBmpWidth;
  SrcBmpHeight := MeMain.ImageBmpHeight;
  RotBmpWidth := Image3D.Picture.Bitmap.Width;
  RotBmpHeight := Image3D.Picture.Bitmap.Height;
  PalMax_1 := MeMain.PalMax - 1;
  RgbQ := SrcBmp.Palette^[0];
  for RotBmpY := 0 to RotBmpHeight - 1 do begin
    pDstLine := Image3D.Picture.Bitmap.ScanLine[RotBmpY];
    for RotBmpX := 0 to RotBmpWidth - 1 do
      pDstLine^[RotBmpX] := RgbQ;
  end;
  oCx := SrcBmpWidth - 1;
  dCx := -1;
  if ((0 < ThetaZ) and (ThetaZ < 180)) then begin
    oCx := 0;
    dCx := +1;
  end;
  oCy := SrcBmpHeight - 1;
  dCy := -1;
  if ((-90 < ThetaZ) and (ThetaZ < 90)) then begin
    oCy := 0;
    dCy := +1;
  end;
  SinX := Sin(Rad(1.0 * ThetaX));
  CosX := Cos(Rad(1.0 * ThetaX));
  SinZ := Sin(Rad(1.0 * ThetaZ));
  CosZ := Cos(Rad(1.0 * ThetaZ));
  for TmpI := 0 to RotBmpWidth - 1 do
    DrawnSy[TmpI] := 0;
  MaxSz := Round(MaxHeight * CosX);
  if (DepthDir = DepthU) then begin
    for TmpI := 0 to MaxSz do
      IdxCz[TmpI] := MulDiv(PalMax_1, TmpI + 1, MaxSz + 1);
    for TmpI := 0 to PalMax_1 do
      IdxSz[TmpI] := MulDiv(MaxSz, TmpI, PalMax_1);
  end
  else begin
    for TmpI := 0 to MaxSz do
      IdxCz[TmpI] := PalMax_1 - MulDiv(PalMax_1, TmpI + 1, MaxSz + 1);
    for TmpI := 0 to PalMax_1 do
      IdxSz[TmpI] := MaxSz - MulDiv(MaxSz, TmpI, PalMax_1);
  end;
  OfsCx := -SrcBmpWidth div 2;
  OfsCy := -SrcBmpHeight div 2;
  sC0x := (RotBmpWidth div 2) + Round((OfsCx * CosZ) - (OfsCy * SinZ));
  sC0y := ((RotBmpHeight - Round(MaxHeight * CosX)) div 2) + Round(((OfsCx * SinZ) + (OfsCy * CosZ)) * SinX);
  for Cx := 0 to SrcBmpWidth - 1 do begin
    sCy0x[Cx] := Cx * CosZ;
    sCy0y[Cx] := Cx * SinZ * SinX;
  end;
  Cy := oCy;
  repeat
    Cx := oCx;
    sCx0x := sC0x - (Cy * SinZ);
    sCx0y := sC0y + (Cy * CosZ * SinX);
    repeat
      Sx := Round(sCx0x + sCy0x[Cx]);
      BottomSy := Round(sCx0y + sCy0y[Cx]);
      Cz := SrcBmp.Pixels^[Cy, Cx];
      RgbQ := SrcBmp.Palette^[Cz];
      dTopSy := IdxSz[Cz];
      if (DrawnSy[Sx] < (BottomSy + dTopSy)) then begin
        dBottomSy := 0;
        if (BottomSy <= DrawnSy[Sx]) then
          dBottomSy := DrawnSy[Sx] - BottomSy + 1;
        DrawnSy[Sx] := BottomSy + dTopSy;
        RotBmpX := Sx;
        RotBmpY := RotBmpHeight - 1 - (BottomSy + dBottomSy);
        if (dBottomSy = (dTopSy - 1)) then begin
          PRgbQuadArray(Image3D.Picture.Bitmap.ScanLine[RotBmpY])^[RotBmpX] := RgbQ;
          Dec(RotBmpY);
        end
        else begin
          for TmpI := dBottomSy to dTopSy - 1 do begin
            PRgbQuadArray(Image3D.Picture.Bitmap.ScanLine[RotBmpY])^[RotBmpX] := SrcBmp.Palette^[IdxCz[TmpI]];
            Dec(RotBmpY);
          end;
        end;
        PRgbQuadArray(Image3D.Picture.Bitmap.ScanLine[RotBmpY])^[RotBmpX] := RgbQ;
        Dec(RotBmpY);
        PRgbQuadArray(Image3D.Picture.Bitmap.ScanLine[RotBmpY])^[RotBmpX] := RgbQ;
      end;
      Inc(Cx, dCx);
    until ((Cx < 0) or (SrcBmpWidth <= Cx));
    Inc(Cy, dCy);
  until ((Cy < 0) or (SrcBmpHeight <= Cy));
  Image3D.Picture.Bitmap.Canvas.UnLock;
end;

procedure TView3DDlg.DrawRotBmp;
begin
  Image3D.Invalidate;
end;

procedure TView3DDlg.SrcBmpChanged;
begin
  SrcBmp.Copy(FormMain.ImageBmp256[CurrentMap]);
  MakeRotBmp;
  DrawRotBmp;
end;

function TView3DDlg.RotThetaX(_Dir: RotDirList): Integer;
var
  NextThetaX: Integer;
begin
  NextThetaX := ThetaX;
  case _Dir of
    RotU: NextThetaX := ThetaX - ThetaStep;
    RotD: NextThetaX := ThetaX + ThetaStep;
  end;
  if (NextThetaX <  0) then NextThetaX :=  0;
  if (90 < NextThetaX) then NextThetaX := 90;
  Result := NextThetaX;
end;

function TView3DDlg.RotThetaZ(_Dir: RotDirList): Integer;
var
  NextThetaZ: Integer;
begin
  NextThetaZ := ThetaZ;
  case _Dir of
    RotL: NextThetaZ := ThetaZ - ThetaStep;
    RotR: NextThetaZ := ThetaZ + ThetaStep;
  end;
  if (NextThetaZ < -180) then NextThetaZ := NextThetaZ + 360;
  if ( 180 < NextThetaZ) then NextThetaZ := NextThetaZ - 360;
  Result := NextThetaZ;
end;

procedure TView3DDlg.SpeedButtonRotClick(Sender: TObject);
begin
  if (FormMain.MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  RotDir := RotN;
  if (Sender = SpeedButtonRotU) then RotDir := RotU
  else if (Sender = SpeedButtonRotR) then RotDir := RotR
  else if (Sender = SpeedButtonRotL) then RotDir := RotL
  else if (Sender = SpeedButtonRotD) then RotDir := RotD;
  case RotDir of
    RotU, RotD : ThetaX := RotThetaX(RotDir);
    RotL, RotR : ThetaZ := RotThetaZ(RotDir);
  end;
  MakeRotBmp;
  DrawRotBmp;
end;

procedure TView3DDlg.SpeedButtonDepthClick(Sender: TObject);
begin
  if (FormMain.MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
  if (Sender = SpeedButtonDepthU) then DepthDir := DepthU
  else if (Sender = SpeedButtonDepthD) then DepthDir := DepthD;
  RotDir := RotN;
  MakeRotBmp;
  DrawRotBmp;
end;

procedure TView3DDlg.ClickToHelp(Sender: TObject);
begin
  if (FormMain.MainMode = mmHelpMode) then begin
    HelpModeDlg.DisplayHelp(Sender);
    Exit;
  end;
end;

end.

